from datetime import timedelta, datetime

from django.contrib.auth.decorators import permission_required

from staff.lib.decorators import responding_xlsx
from staff.lib.mongodb import mongo
from staff.person.models import Staff


@permission_required('django_intranet_stuff.can_view_gaps_with_countries')
@responding_xlsx(prefix='gaps')
def gaps_with_countries(request):
    all_gaps = mongo.db['gaps'].find({
        'master_issue': {'$exists': True},
        'countries_to_visit': {'$exists': True},
        'workflow': 'vacation',
        'state': {'$ne': 'canceled'},
        'created_at': {'$gte': datetime.now() - timedelta(days=180)},
    })

    all_gaps = [g for g in all_gaps if g['countries_to_visit']]
    all_logins = {gap['person_login'] for gap in all_gaps}
    persons = {
        person.login: person
        for person in Staff.objects.select_related('office', 'organization').filter(login__in=all_logins)
    }

    def generate_gaps():
        yield [
            'Логин',
            'Фамилия',
            'Имя',
            'Отчество',
            'Тикет',
            'Офис',
            'Организация',
            'С',
            'По',
        ]

        for gap in all_gaps:
            pers = persons[gap['person_login']]
            yield [
                pers.login,
                pers.last_name,
                pers.first_name,
                pers.middle_name,
                gap['master_issue'],
                pers.office.name,
                pers.organization and pers.organization.name,
                gap['date_from'].date().isoformat(),
                (gap['date_to'].date() - timedelta(days=1)).isoformat(),
            ] + [z for z in gap['countries_to_visit']]

    return [['gaps', generate_gaps()]]
