from datetime import datetime, timedelta
import dateutil.parser
from pytz import timezone
import logging

from django.conf import settings
from django.views.decorators.http import require_GET
from django.http import HttpResponseNotFound, HttpResponseForbidden, HttpResponseServerError
from django.core.urlresolvers import reverse

from staff.lib.decorators import responding_json, available_for_external

from staff.gap.controllers.gap import GapCtl, GapQueryBuilder, PeriodicGapCtl
from staff.gap.controllers.gap_settings import GapSettingsCtl
from staff.gap.controllers.utils import (
    collect_confirmed_by,
    full_day_dates,
    gap_dates_from_utc,
    get_universal_person,
    is_chief_of,
)
from staff.gap.workflows.office_work.utils import get_office_work_extra
from staff.gap.exceptions import GapError
from staff.gap.permissions import can_see_all_gaps, can_see_gap, get_accessible_logins
from staff.gap.views.vacation_views import find_pdf_template
from staff.gap.workflows.absence.workflow import AbsenceWorkflow
from staff.gap.workflows.choices import GAP_STATES as GS, LIMITED_STATES
from staff.gap.workflows.decorators import valid_gap_id
from staff.gap.workflows.utils import find_workflow
from staff.gap.utils import show_edit_button, show_delete_button

logger = logging.getLogger('staff.gap.views.gaps_by_day_views')


@responding_json
@require_GET
@available_for_external
def gaps_by_day(request, **kwargs):
    observer = request.user

    login = kwargs.get('login') or request.GET.get('login', observer.get_profile().login)

    if not get_accessible_logins(observer.get_profile(), [login]):
        logger.info('%s has no access to %s gap', observer, login)
        return HttpResponseForbidden()

    person = get_universal_person(person_login=login)

    if not person:
        return HttpResponseNotFound()

    day_date = kwargs.get('day_date') or (
        dateutil.parser.parse(request.GET.get('day_date'))
        .replace(
            hour=0,
            minute=0,
            second=0,
            tzinfo=None,
        )
    )

    try:
        return _get_gaps_by_day(
            observer=observer,
            person=person,
            day_date=day_date,
            requested_gap_id=kwargs.get('requested_gap_id'),
        )
    except Exception:
        logger.exception('Error trying to get gaps by day')
        return HttpResponseServerError()


@responding_json
@require_GET
@available_for_external
@valid_gap_id
def gaps_by_id(request, gap_id):
    observer = request.user

    try:
        gap_id = int(gap_id)
        try:
            gap = GapCtl().find_gap_by_id(gap_id)
        except GapError:
            return HttpResponseNotFound()

        person = get_universal_person(person_id=gap['person_id'])

        can_see_person = get_accessible_logins(observer.get_profile(), [person['login']])
        if not can_see_person or not can_see_gap(gap, observer, person):
            logger.info('%s has no access to %s gap', observer, person['login'])
            return HttpResponseForbidden()

        if gap['state'] == GS.CANCELED:
            return {
                'redirect': {
                    'login': gap['person_login'],
                    'workflow': gap['workflow'],
                    'date_from': gap['date_from'],
                    'date_to': gap['date_to'],
                    'page': 1,
                    'limit': 10,
                },
                'person_login': gap['person_login'],
                'day_date': _day_from_gap(gap),
            }

        return _get_gaps_by_day(
            observer=observer,
            person=person,
            day_date=_day_from_gap(gap),
            requested_gap_id=gap_id,
        )
    except Exception:
        logger.exception('Error trying to get gaps by ID')
        return HttpResponseServerError()


def _show_confirm_button_for_gap(observer, gap, person_login):
    if gap['state'] != GS.NEW:
        return False

    if gap['workflow'] == 'absence':
        workflow = AbsenceWorkflow(gap=gap)
        return (
            workflow.is_long_absence_and_needs_approval()
            and workflow.person_can_confirm_gap(observer.get_profile().login)
        )

    if gap['workflow'] == 'vacation':
        return observer.is_superuser or is_chief_of(observer.get_profile().login, person_login)

    return False


def _get_gaps_by_day(observer, person, day_date, requested_gap_id=None):
    person_login = person['login']
    observer_tz = timezone(observer.get_profile().tz)

    date_from = day_date + timedelta(seconds=1)
    date_to = day_date + timedelta(hours=23, minutes=59, seconds=59)

    can_see_all = can_see_all_gaps(observer, [person_login])

    result = {
        'gaps': [],
        'person_login': person_login,
        'day_date': day_date,
    }
    # STAFF-17109
    if person_login == settings.LOGIN_TIGRAN and observer.get_profile().login != settings.LOGIN_TIGRAN:
        return result

    gs_ctl = GapSettingsCtl()
    _gaps = list(
        _gaps_by_dates(can_see_all, person_login, date_from, date_to, observer_tz)
    )
    confirmed_by = collect_confirmed_by(_gaps)
    gaps = []
    periodic_gap_ids = [_gap['periodic_gap_id'] for _gap in _gaps if 'periodic_gap_id' in _gap]
    _periodic_gaps = {
        periodic_gap.pop('id'): periodic_gap
        for periodic_gap in periodic_gaps_by_ids(periodic_gap_ids)
        if periodic_gap
    }
    for gap in _gaps:
        gap['show_confirm_button'] = _show_confirm_button_for_gap(observer, gap, person_login)
        if gap.get('periodic_gap_id'):
            periodic_gap_id = gap['periodic_gap_id']
            gap.update(_periodic_gaps[periodic_gap_id])
        gaps.append(_prepare_gap(gap, person, gs_ctl, observer, confirmed_by))

    result['gaps'] = gaps
    if requested_gap_id is not None:
        result['requested_gap_id'] = requested_gap_id

    return result


def _day_from_gap(gap):
    date_from = gap['date_from']
    date_to = gap['date_to']
    now = datetime.now()

    if date_to < now:
        if gap['full_day']:
            date_to = date_to - timedelta(days=1)
        return date_to.replace(hour=0, minute=0, second=0)
    elif date_from > now:
        return date_from.replace(hour=0, minute=0, second=0)
    else:
        return now.replace(hour=0, minute=0, second=0)


def _gaps_by_dates(can_see_all, login, date_from, date_to, observer_tz):
    gqb = (
        GapQueryBuilder()
        .person_login(login)
        .dates_not_strict_with_tz(date_from, date_to, observer_tz)
    )

    if not can_see_all:
        gqb.workflows_states(LIMITED_STATES)
    else:
        gqb.op_ne('state', GS.CANCELED)

    fields = [
        'id',
        'workflow',
        'date_from',
        'date_to',
        'comment',
        'person_login',
        'state',
        'full_day',
        'is_selfpaid',
        'is_covid',
        'work_in_absence',
        'person_id',
        'confirmed_by_id',
        'duration',
        'need_approval',
        'master_issue',
        'service_slug',
        'service_name',
        'shift_id',
        'role_on_duty',
        'place',
        'office',
        'room',
        'table',
        'periodic_gap_id',
        'mandatory',
        'vacation_updated',
        'deadline',
    ]

    return GapCtl().find_gaps(query=gqb.query(), fields=fields)


def periodic_gaps_by_ids(periodic_gap_ids):
    query_builder = (
        GapQueryBuilder()
        .op_in('id', periodic_gap_ids)
    )
    fields = [
        'id',
        'periodic_type',
        'period',
        'periodic_date_to',
        'periodic_map_weekdays',
    ]
    return PeriodicGapCtl().find_gaps(query=query_builder.query(), fields=fields)


def _show_confirm_status(gap):
    if gap['workflow'] in ('vacation', 'conference_trip', 'trip'):
        return True

    return gap['workflow'] == 'absence' and AbsenceWorkflow(gap=gap).is_long_absence_and_needs_approval()


def _prepare_gap(gap, person, gs_ctl, observer, confirmed_by):
    observer_tz = timezone(observer.get_profile().tz)
    gap_id = gap['id']
    workflow = gap['workflow']

    gap['show_confirm_status'] = _show_confirm_status(gap)
    gap['show_delete_button'] = show_delete_button(observer, gap)
    gap['show_edit_button'] = show_edit_button(observer, gap)
    gap['show_external_link_button'] = workflow == 'duty'

    if gap.get('confirmed_by_id'):
        gap['confirmed_by'] = confirmed_by.get(gap['confirmed_by_id'])
        del gap['confirmed_by_id']

    if workflow == 'office_work':
        gap = get_office_work_extra(gap)

    if workflow in ['vacation', 'maternity']:
        if 'is_selfpaid' in gap and gap['is_selfpaid']:
            tag = '%s_selfpaid' % workflow
        else:
            tag = workflow

        vacation_file = gs_ctl.find_workflow_file(person['organization_id'], gap['workflow'], tag)

        if not vacation_file:
            template = find_pdf_template(person, workflow, tag)
            if template:
                vacation_file = reverse('gap:vacation-file', kwargs={'gap_id': gap_id})

        if vacation_file:
            gap['vacation_file'] = vacation_file

    if workflow == 'illness':
        gap['is_covid'] = gap.get('is_covid', False)
    elif 'is_covid' in gap:
        del gap['is_covid']

    gap['workflow'] = find_workflow(gap['workflow']).ui_key
    gap['gap_view_url'] = reverse('gap:gaps-by-id', kwargs={'gap_id': gap_id})

    if workflow == 'duty':
        gap['work_in_absence'] = False
    else:
        gap['gap_edit_url'] = reverse('gap:edit-gap', kwargs={
            'login': gap['person_login'],
            'gap_id': gap_id,
        })

    del gap['person_login']

    if gap['full_day']:
        full_day_dates(gap, -1)
    else:
        gap_dates_from_utc(gap, observer_tz)

    return gap
