from pytz import timezone

from django.views.decorators.http import require_GET
from django.http import (
    HttpResponseBadRequest,
    HttpResponseServerError,
)

from collections import defaultdict
import dateutil.parser
import pymongo

from staff.lib.decorators import responding_json, available_for_external
from staff.lib.calendar import get_holidays
from staff.person.models import Staff
from staff.gap.events_binder import EventsBinder
from staff.gap.controllers.gap import (
    GapCtl,
    GapQueryBuilder,
)
from staff.gap.workflows.choices import (
    GAP_STATES as GS,
    LIMITED_STATES,
)
from staff.gap.workflows.utils import workflows_for_ui
from staff.gap.controllers.utils import (
    get_short_persons_with_department,
    is_gap_partial,
    get_gap_days,
    gap_dates_from_utc,
)
from staff.gap.permissions import divide_by_perms, get_readable_accessible_logins

import logging
logger = logging.getLogger('staff.gap.views.gaps_calendar_view')


@responding_json
@require_GET
@available_for_external
def gaps_calendar(request):
    observer = request.user
    observer_tz = timezone(observer.get_profile().tz)

    logins = get_readable_accessible_logins(observer.get_profile(), request.GET.getlist('l', []))
    date_from = dateutil.parser.parse(request.GET.get('date_from')).replace(tzinfo=None)
    date_to = dateutil.parser.parse(request.GET.get('date_to')).replace(tzinfo=None)

    if not logins:
        return {}

    persons_geo_ids = _get_persons_geo_ids(logins)
    holidays = _holidays_to_shift(
        date_from.date(),
        get_holidays(date_from, date_to, set(persons_geo_ids.values())),
    )

    try:
        persons = get_short_persons_with_department(logins)

        if persons.count() != len(logins):
            return HttpResponseBadRequest()

        can_see_all, can_see_normal = divide_by_perms(observer, logins)
        gaps = _get_calendar(can_see_all, can_see_normal, date_from, date_to)

        _persons = dict()

        for person in persons:
            login = person['login']

            binder = EventsBinder(start_key='date_from', end_key='date_to')

            _gaps = [_prepare_gap(gap, date_from, binder, observer_tz)
                     for gap in gaps[login]]

            # не можем удалять эти поля, пока биндер всё не сбиндил
            for gap in _gaps:
                del gap['date_from']
                del gap['date_to']
                del gap['workflow']
                del gap['state']
                del gap['full_day']

            _persons[login] = {
                'geo_id': persons_geo_ids[login] or 225,
                'gaps': _gaps,
                'work_mode': person['work_mode'],
            }

    except Exception:
        logger.exception('Error getting gaps calendar')
        return HttpResponseServerError()

    return {
        'workflows': workflows_for_ui(),
        'persons': _persons,
        'holidays': holidays,
    }


def _get_persons_geo_ids(logins):
    return dict(
        Staff.objects
        .filter(login__in=logins)
        .values_list(
            'login',
            'office__city__geo_id',
        )
    )


def _prepare_gap(gap, date_from, binder, observer_tz):
    gap_dates_from_utc(gap, observer_tz)
    _date_from = gap['date_from']
    _date_to = gap['date_to']
    gap['id'] = gap['id']
    gap['s'] = (_date_from.date() - date_from.date()).days               # shift
    gap['l'] = get_gap_days(
        _date_from, _date_to, gap['full_day'], partial_as_full=True)     # length
    gap['k'] = gap['workflow']                                           # key
    gap['p'] = is_gap_partial(_date_from, _date_to)
    gap['show_lock'] = gap.get('mandatory', False)

    binder.bind(gap)

    return gap


def _holidays_to_shift(date_from, holidays):
    return {
        geo_id: [
            {
                's': (day['date'] - date_from).days,
                't': day['day-type'],
                'd': day.get('holiday-name', ''),
            } for day in days]
        for geo_id, days in holidays.items()
    }


def _get_calendar(can_see_all, can_see_normal, date_from, date_to):
    query = GapQueryBuilder().person_logins(can_see_all + can_see_normal).dates_not_strict(date_from, date_to)

    all_query = (
        GapQueryBuilder()
        .person_logins(can_see_all)
        .op_ne('state', GS.CANCELED)
        .query()
    )

    normal_query = (
        GapQueryBuilder()
        .person_logins(can_see_normal)
        .workflows_states(LIMITED_STATES)
        .query()
    )

    query.op_or([all_query, normal_query])

    fields = [
        'id',
        'person_login',
        'date_from',
        'date_to',
        'workflow',
        'state',
        'full_day',
        'mandatory',
    ]

    sorting = [
        ('person_id', pymongo.ASCENDING),
        ('date_from', pymongo.ASCENDING),
    ]

    result = defaultdict(list)

    qs = GapCtl().find_gaps(query=query.query(), fields=fields, sorting=sorting)

    for gap in qs:
        result[gap['person_login']].append(gap)
        del gap['person_login']

    return result
