from pytz import timezone

from django.views.decorators.http import require_GET
from django.http import (
    HttpResponseBadRequest,
    HttpResponseForbidden,
    HttpResponseServerError,
)
from django.core.urlresolvers import reverse

from staff.person.models import Staff

from staff.lib.decorators import responding_json
from staff.lib.utils.qs_values import localize
from staff.person_profile.controllers.subordinates import (
    direct_subordinates_q,
    all_subordinates_q,
)

from staff.gap.controllers.unconfirmed import (
    get_unconfirmed_gaps,
    get_unconfirmed_gaps_counters,
)
from staff.gap.workflows.utils import workflows_for_ui
from staff.gap.controllers.utils import (
    get_gap_days,
    full_day_dates,
    gap_dates_from_utc,
)

import logging
logger = logging.getLogger('staff.gap.views.unconfirmed_gaps_view')


@responding_json
@require_GET
def unconfirmed_gaps(request):
    observer = request.user

    try:
        limit = int(request.GET.get('limit', 10))
        page = int(request.GET.get('page', 1))
        all_subordinates = bool(int(request.GET.get('all', 0)))

        if limit < 1 or limit > 100:
            limit = 10
        if page < 1:
            page = 1
    except (TypeError, ValueError):
        return HttpResponseBadRequest()

    observer_id = observer.get_profile().id
    observer_tz = timezone(observer.get_profile().tz)

    try:
        all_subordinates_ids = list(
            Staff.objects
            .filter(all_subordinates_q(observer_id))
            .values_list('id', flat=True)
        )
        if not all_subordinates_ids:
            return HttpResponseForbidden()
        direct_subordinates_ids = None
        if all_subordinates:
            s_ids = all_subordinates_ids
        else:
            direct_subordinates_ids = s_ids = list(
                Staff.objects
                .filter(direct_subordinates_q(observer_id))
                .values_list('id', flat=True)
            )

        gaps = list(get_unconfirmed_gaps(s_ids, limit, page - 1))

        persons_data = (
            Staff.objects.filter(
                id__in=set([gap['person_id'] for gap in gaps])
            ).values(
                'id',
                'login',
                'first_name',
                'last_name',
                'first_name_en',
                'last_name_en',
            )
        )

        persons_dict = {p['id']: localize(p) for p in persons_data}

        for gap in gaps:
            gap['person'] = persons_dict[gap['person_id']]

            date_from = gap['date_from']
            date_to = gap['date_to']
            gap['days'] = get_gap_days(date_from, date_to, gap['full_day'])
            if gap['full_day']:
                full_day_dates(gap, -1)
            else:
                gap_dates_from_utc(gap, observer_tz)

            gap['confirm_url'] = reverse('gap:confirm-gap', kwargs={'gap_id': gap['id']})

            del gap['person_login']
            del gap['person_id']

        unconfirmed_counters = get_unconfirmed_gaps_counters(
            observer_id,
            direct_subordinates_ids=direct_subordinates_ids,
            all_subordinates_ids=all_subordinates_ids,
        )

        if all_subordinates:
            total = unconfirmed_counters['all']
        else:
            total = unconfirmed_counters['direct']

        return {
            'workflows': workflows_for_ui(),
            'gaps': gaps,
            'limit': limit,
            'page': page,
            'all': all_subordinates,
            'total': total,
            'unconfirmed': {
                "enabled": True,
                "counters": unconfirmed_counters,
            }
        }

    except Exception:
        logger.exception('Error getting unconfirmed gaps')
        return HttpResponseServerError()
