from datetime import timedelta

from django.core.exceptions import ValidationError

import sform

from staff.gap.controllers.utils import replace_useless_smt
from staff.gap.api.forms import WEEKDAYS, PeriodicType
from staff.gap.workflows.choices import FRONTEND_WORKFLOWS

DATETIME_INPUT_FORMATS = (
    '%Y-%m-%dT%H:%M:%S.%fZ',  # '2006-10-25T14:30:59.000Z'
    '%Y-%m-%dT%H:%M:%S',      # '2006-10-25T14:30:59'
    '%Y-%m-%dT%H:%M',         # '2006-10-25T14:30'
    '%Y-%m-%d',               # '2006-10-25'
)


class CaptionedCharField(sform.CharField):
    def data_as_dict(self, *args, **kwargs):
        field_dict = super(CaptionedCharField, self).data_as_dict(*args, **kwargs)
        field_dict['caption'] = field_dict['value'][1]
        field_dict['value'] = field_dict['value'][0]
        return field_dict


class ToNotifyForm(sform.SForm):
    email = CaptionedCharField(state=sform.REQUIRED)

    def clean_email(self, value):
        return value.strip() if value else value


class BaseGapForm(sform.SForm):
    valid_gap_length_days = None
    invalid_gap_length_days_message = 'gap length in days is greater than you can create'
    workflow = sform.ChoiceField(
        choices=FRONTEND_WORKFLOWS,
        state=sform.READONLY,
        empty_label='—',
    )
    full_day = sform.BooleanField(state=sform.NORMAL, default=False)
    date_from = sform.DateTimeField(
        state=sform.REQUIRED,
        input_formats=DATETIME_INPUT_FORMATS,
    )
    date_to = sform.DateTimeField(
        state=sform.REQUIRED,
        input_formats=DATETIME_INPUT_FORMATS,
    )
    work_in_absence = sform.BooleanField(state=sform.REQUIRED, default=False)

    def clean_date_from(self, value):
        return replace_useless_smt(value)

    def clean_date_to(self, value):
        date_from = self.cleaned_data['date_from']
        date_to = value
        full_day = self.cleaned_data['full_day']

        if date_to < date_from:
            raise ValidationError(
                'date_to is less than date_from',
                code='invalid',
            )

        if date_to == date_from and not full_day:
            raise ValidationError(
                'date_to is equal to date_from and it is not full_day',
                code='invalid',
            )

        if self.valid_gap_length_days is not None:
            days = (date_to - date_from).days
            if full_day:
                days += 1
            if days > self.valid_gap_length_days:
                raise ValidationError(
                    self.invalid_gap_length_days_message,
                    code='days_limit_reached'
                )

        return replace_useless_smt(date_to)

    def make_all_readonly(self, except_fields=None):
        except_fields = except_fields or []
        for name in self.fields_state:
            if name not in except_fields:
                self.fields_state[name] = sform.READONLY

    def make_field_readonly(self, field_name):
        if field_name in self.fields_state:
            self.fields_state[field_name] = sform.READONLY


class BasePeriodicGapForm(BaseGapForm):
    is_periodic_gap = sform.BooleanField(
        state=sform.REQUIRED,
        default=False,
    )
    periodic_date_to = sform.DateTimeField(
        state=sform.NORMAL,
        input_formats=DATETIME_INPUT_FORMATS,
    )
    periodic_type = sform.ChoiceField(
        choices=PeriodicType.choices(),
        default=PeriodicType.WEEK.value,
    )
    period = sform.IntegerField(state=sform.NORMAL, default=1)
    periodic_map_weekdays = sform.MultipleChoiceField(
        choices=WEEKDAYS.choices(),
        state=sform.NORMAL,
    )

    def clean_periodic_date_to(self, value):
        is_periodic_gap = self.cleaned_data['is_periodic_gap']
        if not is_periodic_gap:
            return None
        if value is None:
            raise ValidationError(
                'periodic_date_to is required',
                code='periodic_date_to_empty',
            )
        date_to = self.cleaned_data['date_to']
        periodic_date_to = value

        if date_to > periodic_date_to:
            raise ValidationError(
                'periodic_date_to is less than date_to',
                code='invalid',
            )

        if periodic_date_to > date_to + timedelta(days=31 * 3):
            raise ValidationError(
                'periodic_date_to is more than date_to + 3 month',
                code='invalid',
            )

        return replace_useless_smt(periodic_date_to)

    def clean_periodic_map_weekdays(self, value):
        is_periodic_gap = self.cleaned_data['is_periodic_gap']
        if not is_periodic_gap:
            return None
        if not value:
            raise ValidationError(
                'periodic_map_weekdays is required in periodic gap',
                code='invalid',
            )
        return sorted(value)
