import datetime

from typing import Dict, List, Set
from operator import itemgetter
from itertools import groupby

from django.conf import settings

from staff.groups.models import Group, GROUP_TYPE_CHOICES
from staff.lib import requests, tvm2
from staff.lib.utils.date import parse_datetime

from staff.gap.controllers.gap import (
    GapCtl,
    GapQueryBuilder,
)
from staff.gap.workflows.duty.workflow import DutyWorkflow
import logging

logger = logging.getLogger(__name__)


ABC_FIELDS = [
    'id',
    'person.id',
    'person.login',
    'schedule.id',
    'schedule.name',
    'is_approved',
    'start_datetime',
    'end_datetime',
    'replaces.id',
    'replaces.person.id',
    'replaces.person.login',
    'replaces.schedule.id',
    'replaces.schedule.name',
    'replaces.start',
    'replaces.end',
    'replaces.is_approved',
]


def sync_from_abc(service_id: int = None) -> None:
    schedules = fetch_schedules(service_id)
    show_schedules: set = {sch_id for sch_id, sch in schedules.items() if sch['show_in_staff']}

    all_shifts = get_shifts_and_replaces(service_id, schedules)
    all_shifts.sort(key=itemgetter('service_slug'))

    sync_service_slugs = {shift['service_slug'] for shift in all_shifts}
    all_service_slugs = Group.objects.filter(type=GROUP_TYPE_CHOICES.SERVICE).values_list('code', flat=True)
    if service_id is not None:
        all_service_slugs = all_service_slugs.filter(service_id=service_id)

    removed_service_slugs = set(all_service_slugs) - sync_service_slugs

    removed_gaps_query = (
        GapQueryBuilder()
        .workflow('duty')
        .op_in('service_slug', list(removed_service_slugs))
    )
    for removed_gap in GapCtl().find_gaps(removed_gaps_query.query(), fields=['id']):
        _cancel_gap(removed_gap['id'])

    for service_slug, service_shifts in groupby(all_shifts, key=itemgetter('service_slug')):
        service_shifts = {sh['id']: sh for sh in service_shifts}
        shifts_to_update: Set[int] = {
            sh_id for sh_id, sh in service_shifts.items()
            if sh['schedule']['id'] in show_schedules
        }

        shifts_with_replaces_ids = get_shifts_with_replaces_ids(service_shifts, shifts_to_update)

        logger.info('processing service %s: with %s shifts', service_slug, len(service_shifts))
        existing_gaps_query = (
            GapQueryBuilder()
            .workflow('duty')
            .op_in('shift_id', list(shifts_to_update - shifts_with_replaces_ids))
        )
        existing_gaps_with_replaces_query = (
            GapQueryBuilder()
            .workflow('duty')
            .op_in('shift_id', list(shifts_with_replaces_ids))
        )
        extra_gaps_query = (
            GapQueryBuilder()
            .workflow('duty')
            .op_eq('service_slug', service_slug)
            .op_not('shift_id', {'$in': list(shifts_to_update)})
        )

        processed_shift_ids = []
        is_approved_shift_ids = set()

        # обновляем существующие гэпы без замен или сами замены (и отменяем повторы если они есть)
        for existing_gap in GapCtl().find_gaps(existing_gaps_query.query(), fields=['id', 'shift_id']):
            shift_data = service_shifts[existing_gap['shift_id']]
            is_approved = shift_data['is_approved']
            if existing_gap['shift_id'] in is_approved_shift_ids:
                is_approved = False
            if is_approved:
                is_approved_shift_ids.add(shift_data['id'])
            _process_existing_gap(
                gap_id=existing_gap['id'],
                gap_data=shift_to_gap(shift_data, start=shift_data['start'], end=shift_data['end']),
                approved=is_approved,
            )
            processed_shift_ids.append(shift_data['id'])
        logger.info('processed %s shifts, %s approved_shifts', len(processed_shift_ids), len(is_approved_shift_ids))

        # создаем или изменяем гепы с заменой(полностью удаляем и заново создаем)
        for extra_gap in GapCtl().find_gaps(existing_gaps_with_replaces_query.query(), fields=['id']):
            _cancel_gap(extra_gap['id'])

        # создаём недостающие гэпы
        for shift_id, shift_data in service_shifts.items():
            if shift_id not in shifts_to_update:
                continue
            if shift_id in processed_shift_ids:
                continue
            if not shift_data['is_approved']:
                continue
            _create_new_gaps(shift_data)

        # удаляем лишние гэпы
        for extra_gap in GapCtl().find_gaps(extra_gaps_query.query(), fields=['id']):
            _cancel_gap(extra_gap['id'])


def get_shifts_with_replaces_ids(service_shifts: dict, shifts_to_update: set):
    shifts_with_replaces_ids: Set[int] = set()

    for sh_id, sh in service_shifts.items():
        replaces = sh.get('replaces', [])
        if sh_id not in shifts_to_update:
            continue
        approve_replace = False
        for _ in replaces:
            approve_replace = True
            # поправить после ABC-9269
            # if replace['is_approved']:
            #    approve_replace = True
            #     break

        if approve_replace:
            shifts_with_replaces_ids.add(sh_id)

    return shifts_with_replaces_ids


def get_shifts_and_replaces(service_id: int, schedules: dict) -> List[dict]:
    all_shifts = []
    for sh in _fetch_shifts(service_id):
        if sh['schedule']['id'] not in schedules:
            logger.info('Unknown schedule %s', sh)
            continue
        all_shifts.append(_extend_shift(sh, schedules))
        all_shifts += [_extend_shift(shift_replace, schedules, is_approved=True) for shift_replace in sh['replaces']]

    return all_shifts


def _extend_shift(shift: dict, schedules_by_id: dict, is_approved: bool = None) -> dict:
    schedule = schedules_by_id[shift['schedule']['id']]
    if is_approved:
        shift['is_approved'] = True
    shift['service_slug'] = schedule['service']['slug']
    shift['service_name'] = schedule['service']['name']['ru'] if schedule['service']['name'] else ''
    shift['role_on_duty'] = schedule['role'] if schedule['role'] and schedule['role']['name']['ru'] else ''

    shift['start'] = parse_datetime(shift['start_datetime'])
    shift['end'] = parse_datetime(shift['end_datetime'])

    return shift


def _process_existing_gap(gap_id: int, gap_data: dict, approved: bool) -> None:
    wf = DutyWorkflow.init_to_modify(modifier_id=settings.ROBOT_STAFF_ID, gap_id=gap_id)
    if approved:
        return wf.edit_gap(gap_data)
    return wf.cancel_gap()


def _create_new_gaps(shift_data: dict):
    person_id = shift_data['person'] and shift_data['person']['id']
    if not person_id:
        return

    wf = DutyWorkflow.init_to_new(modifier_id=settings.ROBOT_STAFF_ID, person_id=person_id)
    replaces = shift_data.get('replaces', [])

    start = shift_data['start']

    for shift_replaces in sorted(replaces, key=lambda replace: replace['start']):
        if start < shift_replaces['start']:
            wf.new_gap(shift_to_gap(shift_data, start=start, end=shift_replaces['start']))
        start = shift_replaces['end']

    if start < shift_data['end']:
        wf.new_gap(shift_to_gap(shift_data, start=start, end=shift_data['end']))


def _cancel_gap(gap_id: int) -> None:
    wf = DutyWorkflow.init_to_modify(modifier_id=settings.ROBOT_STAFF_ID, gap_id=gap_id)
    wf.cancel_gap()


def _fetch_shifts(service_id: int = None) -> List[dict]:
    date_from = datetime.datetime.utcnow().date()
    chunk_size = 7
    num_chunks = 4
    date_to = date_from + datetime.timedelta(days=chunk_size*num_chunks)

    url = f'{settings.ABC_URL}/api/v4/duty/shifts/'
    result = {}
    while date_from < date_to:
        current_date_to = date_from + datetime.timedelta(days=chunk_size)
        query = {
            'fields': ','.join(ABC_FIELDS),
            'date_from': date_from.strftime('%Y-%m-%d'),
            'date_to': current_date_to.strftime('%Y-%m-%d'),
            'with_watcher': 1,
        }
        if service_id:
            query['service'] = service_id

        response = requests.get(
            url=url,
            headers={tvm2.TVM_SERVICE_TICKET_HEADER: tvm2.get_tvm_ticket_by_deploy('abc')},
            params=query,
            timeout=(4, 18),  # 20 секунд -- таймаут в ABC
        )
        for shift in response.json().get('results', []):
            result[shift['id']] = shift
        date_from = current_date_to
    return list(result.values())


def fetch_schedules(service_id: int = None) -> Dict[int, dict]:
    url = f'{settings.ABC_URL}/api/v4/duty/schedules-cursor/'
    fields = ['id', 'service.slug', 'service.name', 'role.name', 'show_in_staff']
    query = {
        'page_size': 100,
        'fields': ','.join(fields),
    }

    if service_id:
        query['service'] = service_id

    schedules = []

    while url:
        response = requests.get(
            url=url,
            headers={tvm2.TVM_SERVICE_TICKET_HEADER: tvm2.get_tvm_ticket_by_deploy('abc')},
            params=query,
            timeout=(4, 8, 16),
        )
        response_data = response.json()
        schedules.extend(response_data.get('results', []))
        url = response_data.get('next')

    return {sch['id']: sch for sch in schedules}


def shift_to_gap(shift_data: dict, start: datetime.datetime, end: datetime.datetime) -> dict:
    person_login = shift_data['person'] and shift_data['person']['login']
    person_id = shift_data['person'] and shift_data['person']['id']

    is_full_day = start.time() == datetime.time(0, 0) and end.time() == datetime.time(0, 0)

    if is_full_day:
        start = start.replace(tzinfo=None)
        end = end.replace(tzinfo=None)

    return {
        'shift_id': shift_data['id'],
        'person_login': person_login,
        'person_id': person_id,
        'date_from': start,
        'date_to': end,
        'service_slug': shift_data['service_slug'],
        'service_name': shift_data['service_name'],
        'role_on_duty': shift_data['role_on_duty'],
        'full_day': is_full_day,
        'work_in_absence': True,
    }
