from django.core.exceptions import ValidationError

import sform

from staff.gap.workflows.base_forms import ToNotifyForm, BasePeriodicGapForm
from staff.gap.workflows.choices import OFFICE_PLACE_STATES
from staff.map.models import Office, Room, Table, ROOM_TYPES


class RoomSuggestField(sform.SuggestField):
    @property
    def type_name(self):
        return 'suggest'

    def structure_as_dict(self, *args, **kwargs):
        field_dict = super().structure_as_dict(*args, **kwargs)
        field_dict['types'] = ['new_room']
        return field_dict


class TableCanBookSuggestField(sform.SuggestField):
    @property
    def type_name(self):
        return 'suggest'

    def structure_as_dict(self, *args, **kwargs):
        field_dict = super().structure_as_dict(*args, **kwargs)
        field_dict['types'] = ['table_can_be_booked']
        return field_dict


class OfficeWorkForm(BasePeriodicGapForm):
    comment = sform.CharField(state=sform.NORMAL, default='', max_length=512)
    to_notify = sform.GridField(sform.FieldsetField(ToNotifyForm))
    place = sform.ChoiceField(
        state=sform.REQUIRED,
        choices=OFFICE_PLACE_STATES,
        default=OFFICE_PLACE_STATES.NOPLACE,
    )
    office = sform.ModelChoiceField(
        state=sform.REQUIRED,
        queryset=Office.objects.filter(intranet_status=1).order_by('name'),
        label_extractor='name',
        empty_label='-',
    )
    room = RoomSuggestField(
        state=sform.NORMAL,
        queryset=Room.objects.filter(room_type=ROOM_TYPES.OFFICE, intranet_status=1),
        label_fields=('num', 'name'),
    )
    coworking = RoomSuggestField(
        state=sform.NORMAL,
        queryset=Room.objects.filter(room_type=ROOM_TYPES.COWORKING, intranet_status=1),
        label_fields=('num', 'name'),
    )
    table = TableCanBookSuggestField(
        state=sform.NORMAL,
        queryset=Table.objects.filter(intranet_status=1),
        label_fields='id',
    )

    def clean_office(self, value):
        if not value:
            raise ValidationError(
                'Office is required',
                code='required',
            )
        return value

    def clean_room(self, value):
        place = self.cleaned_data['place']
        office = self.cleaned_data.get('office')
        room = value

        if not office:
            return

        if place == OFFICE_PLACE_STATES.TEAMROOM:
            if not room:
                raise ValidationError(
                    'Room is required',
                    code='required',
                )

            if not room.floor.office_id == office.pk:
                raise ValidationError(
                    'Wrong room selected',
                    code='invalid_choice',
                )

            return value

    def clean_coworking(self, value):
        place = self.cleaned_data['place']
        office = self.cleaned_data.get('office')
        coworking = value

        if not office:
            return

        if place == OFFICE_PLACE_STATES.COWORKING:
            if not coworking:
                raise ValidationError(
                    'Coworking is required',
                    code='required',
                )

            if not coworking.floor.office_id == office.pk:
                raise ValidationError(
                    'Wrong coworking selected',
                    code='invalid_choice',
                )

        return value

    def clean(self):
        cleaned_data = self.cleaned_data

        for field_name in ('office', 'room', 'coworking', 'table'):
            value = cleaned_data.get(field_name)
            cleaned_data[field_name] = value and getattr(value, 'pk', None)

        return cleaned_data
