from datetime import datetime, timedelta
import logging

from django.db.models import Q

from staff.map.models import TableBook

from staff.gap.controllers.gap import GapQueryBuilder
from staff.gap.controllers.utils import get_chief
from staff.gap.workflows.base_workflow import BasePeriodicWorkflow
from staff.gap.workflows.decorators import allowed_states
from staff.gap.workflows.choices import GAP_STATES as GS


logger = logging.getLogger(__name__)


class OfficeWorkWorkflow(BasePeriodicWorkflow):
    editable_fields = [
        'place',
        'office',
        'room',
        'coworking',
        'table',
    ] + BasePeriodicWorkflow.editable_fields

    workflow = 'office_work'
    ui_key = 'gap.workflow.office_work'
    gap_type = 'office_work'
    verbose_name = 'Работа в офисе'
    verbose_name_en = 'Office work'
    color = '#122faa'

    template_context = {
        'new_gap_head_ru': 'Работа в офисе создана',
        'new_gap_head_en': 'Office work was created',
        'edit_gap_head_ru': 'Работа в офисе изменена',
        'edit_gap_head_en': 'Office work was edited',
        'cancel_gap_head_ru': 'Работа в офисе отменена',
        'cancel_gap_head_en': 'Office work was canceled',
    }

    def _get_approver(self):
        if self.periodic_gap:
            return get_chief(self.periodic_gap['person_login'])
        return get_chief(self.gap['person_login'])

    def _book_table(self, data, gap):
        table_id = data.get('table')

        if not table_id:
            return

        staff_id = self._person_id

        query = Q(table_id=table_id) | Q(staff_id=staff_id)
        query &= Q(date_from__lt=data['date_to'].date()) & Q(date_to__gte=data['date_from'].date())

        if TableBook.objects.filter(query).exists():
            logger.info("There is a conflict when booking table in table_id=%s, staff_id=%s", table_id, staff_id)
            gap['table'] = None
            return

        book = TableBook(
            created_at=datetime.now(),
            modified_at=datetime.now(),
            staff_id=staff_id,
            table_id=table_id,
            date_from=data['date_from'].date(),
            date_to=data['date_to'].date() - timedelta(days=1),
            description=data.get('comment', ''),
            gap_id=gap['id'],
        )
        book.save()

    def _delete_book_table(self):
        gqb = GapQueryBuilder()
        if self.gap:
            gqb.gap_id(self.gap['id'])
        if self.periodic_gap:
            gqb.gaps_with_period(self.periodic_gap['id'])
        if self.gap_from:
            gqb.date_from(self.gap_from['date_from'])
        gap_ids = (gap['id'] for gap in self.gap_ctl.find_gaps(query=gqb.query(), fields=['id']))

        qs = TableBook.objects.filter(gap_id__in=gap_ids)
        qs.delete()

    def new_gap(self, data):
        self.gap = self._create_new_gap(data)

        self._new_gap_email(approver=self._get_approver())

        return self.gap

    def _create_new_gap(self, data):
        _data = data.copy()
        gap = self.gap_ctl.new_gap(self._modifier_id, self._create_gap(_data))
        self._book_table(_data, gap=gap)
        self.gap_ctl.update_gap(self._modifier_id, gap)

        return gap

    def new_periodic_gap(self, data):
        self._create_self_periodic_gap(data)
        self._new_periodic_gap_email(approver=self._get_approver())

        # Возвращаем первый созданный геп, для того что бы фронт мог отрисовать карточку созданного отсутствия
        return self.gap

    def _append_gap(self, data):
        return {
            'place': data.get('place'),
            'office': data.get('office'),
            'room': data.get('room'),
            'coworking': data.get('coworking'),
            'table': data.get('table'),
            'work_in_absence': True,
        }

    def _edit_book_table(self, updated, data):
        if {'table', 'date_from', 'date_to'} & set(updated.keys()):
            self._delete_book_table()
            self._book_table(data, self.gap)

    @allowed_states([GS.NEW])
    def edit_gap(self, data):
        data['work_in_absence'] = True
        updated = self._update_editable(data, self.editable_fields)
        if not updated:
            return

        self._edit_book_table(updated, data)

        self._update()
        if data['is_periodic_gap']:
            return

        self._new_to_notify(updated)

        approver = self._get_approver()
        self._edit_gap_email(updated, approver=approver)

    @allowed_states([GS.NEW])
    def cancel_gap_with_period(self):
        self._set_state_gaps_with_period(GS.CANCELED)
        self._delete_book_table()

        self._update_periodic_gap()
        self._cancel_periodic_gap_email(approver=self._get_approver())

    @allowed_states([GS.NEW])
    def cancel_gap(self):
        self._set_state(GS.CANCELED)
        self._delete_book_table()
        self._update()

        approver = self._get_approver()
        self._cancel_gap_email(approver=approver)
