from datetime import timedelta
from typing import Dict, Any, Optional
import logging

from django.conf import settings
from django.template import Context, Template

from staff.celery_app import app
from staff.lib.auth.utils import get_user_by_uid

from staff.gap.controllers.gap import GapCtl
from staff.gap.controllers.gap_tasks import GapTasks
from staff.gap.controllers.startrek import StartrekCtl
from staff.gap.controllers.templates import GapTemplateContext, get_templates_by_logins
from staff.gap.controllers.utils import get_chiefs

from staff.groups.models import Group

logger = logging.getLogger(__name__)


@app.task
def create_issue_for_gap(gap_id, modifier_id, login, master_issue_field, params):
    gap_ctl = GapCtl()
    gap = gap_ctl.find_gap_by_id(gap_id)
    issue = StartrekCtl(author_login=login).create(**params)
    gap[master_issue_field] = issue['key']
    gap_ctl.update_gap(modifier_id, gap)


def change_gap_ticket_status(gap_id, startrek_ctl, issue_comment, issue_key, tag):
    action = issue_comment[1]
    if action:
        log_info = (action, tag, gap_id, issue_key)
        if action in ['approve', 'cancel', 'reopen', 'will_not_fix']:
            logger.info('Skipping action %s tag %s on gap[%s], issue[%s]', *log_info)
        else:
            logger.info('Performing action %s tag %s on gap[%s], issue[%s]', *log_info)
            startrek_ctl.change_state(action)


@app.task
def comment_issue_for_gap(
    gap_id,
    issue_key,
    issue_comment,
    tag,
    author_login,
    login_to_call,
    reset_dates_params,
    forbidden_statuses,
):
    ctl = StartrekCtl(author_login=author_login, issue_key=issue_key)
    if forbidden_statuses and ctl.issue.status.key in forbidden_statuses:
        logger.info('Skipping comment for gap %s in status %s', gap_id, ctl.issue.status.key)
        return

    ctl.comment(issue_comment[0], [login_to_call] if login_to_call else None)

    change_gap_ticket_status(gap_id, ctl, issue_comment, issue_key, tag)

    if reset_dates_params:
        ctl.issue.update(**reset_dates_params)


class StartrekMixin(object):
    DEFAULT_ISSUE_FIELD = 'master_issue'
    DEFAULT_ISSUE_UNIQUE_PREFIX = 'staff_gap_'

    def _create_issue(self, params, master_issue_field) -> None:
        gap_id = self.gap['id']
        modifier_id = self._modifier_id
        login = get_user_by_uid(self.gap['created_by_uid']).username

        if not master_issue_field:
            master_issue_field = self.DEFAULT_ISSUE_FIELD

        GapTasks.schedule_ordered_task(
            gap_id,
            callable_or_task=create_issue_for_gap,
            kwargs={
                'gap_id': gap_id,
                'modifier_id': modifier_id,
                'login': login,
                'master_issue_field': master_issue_field,
                'params': params,
            },
        )

    def _get_template(self, tag):
        person_login = self.gap['person_login']
        templates_by_logins = get_templates_by_logins(
            [person_login],
            'st_issue',
            self.gap['workflow'],
            tag,
        )

        t_data: Dict[str, Optional[Dict[str, Any]]] = templates_by_logins[person_login]

        if not t_data['template']:
            logger.warning(
                'There are no issue template for gap[%s] and tag "%s"' % (self.gap['id'], tag),
            )
            return None
        return t_data['template']['template']

    def _get_issue_comment(self, tag, gap_diff):
        person_login = self.gap['person_login']
        templates_by_logins = get_templates_by_logins(
            [person_login],
            'st_issue_comment',
            self.gap['workflow'],
            tag,
        )

        t_data = templates_by_logins[person_login]

        if not t_data['template']:
            return None

        comment_template = t_data['template']['template']

        return (
            Template(comment_template['text']).render(self._get_gap_context(gap_diff)),
            comment_template['state_action_id'],
        )

    def _get_gap_context(self, gap_diff):
        return Context(
            GapTemplateContext(self.person, self.modifier, self.gap, gap_diff).context()
        )

    def new_issue(
        self, tag: str,
        append_followers=None,
        master_issue_field=None,
        issue_unique_prefix=None,
    ) -> Dict[str, Any]:
        issue_params = self.prepare_issue_params(
            tag,
            append_followers=append_followers,
            issue_unique_prefix=issue_unique_prefix,
        )

        if not issue_params:
            return self.gap

        self._create_issue(issue_params, master_issue_field)

        return self.gap

    def prepare_issue_params(
        self, tag: str,
        append_followers=None,
        issue_unique_prefix=None,
    ) -> Optional[Dict[str, Any]]:
        template = self._get_template(tag)
        if not template:
            return None

        issue_params = self.get_initial_issue_params()
        issue_params.update(template)

        context = self._get_gap_context({})
        for field in ('summary', 'description'):
            issue_params[field] = Template(template[field]).render(context)

        person_login = self.gap['person_login']
        issue_params['followers'] = [person_login] + (
            append_followers if append_followers else [])

        issue_params['access'] = list({person_login, self.modifier['login']})

        chiefs = [chief['login'] for chief in get_chiefs(person_login)]
        issue_params['access'].extend(chiefs)

        issue_params['access'].extend(self.hrpb_logins)

        date_from = self.gap['date_from']
        date_to = self.gap['date_to']
        if self.gap['full_day']:
            date_to = date_to - timedelta(days=1)
        issue_params['start'] = date_from.date().isoformat()
        issue_params['end'] = date_to.date().isoformat()
        issue_params['employee'] = person_login
        issue_params['department'] = self.person['department__name']
        issue_params['office'] = self.person['office__name']

        if '_id' in self.gap:
            issue_params['unique'] = '{prefix}{uid}'.format(
                prefix=(issue_unique_prefix or self.DEFAULT_ISSUE_UNIQUE_PREFIX),
                uid=str(self.gap['_id']),
            )

        return issue_params

    def get_initial_issue_params(self):
        return {}

    def issue_comment(
        self,
        tag,
        gap_diff=None,
        reset_dates=False,
        forbidden_statuses=None,
        login_to_call=None,
        master_issue_field=None,
    ):
        if not master_issue_field:
            master_issue_field = self.DEFAULT_ISSUE_FIELD

        issue_key = self.gap.get(master_issue_field, None)
        if not issue_key:
            logger.info('No master issue to comment for gap %s', self.gap['id'])
            return

        forbidden_logins = list(Group.objects.filter(url='top_managers').values_list('members__login', flat=True))
        forbidden_logins += settings.NOTIFICATIONS_FORBIDDEN_TO_NOTIFY

        if login_to_call in forbidden_logins:
            login_to_call = None

        exclude_from_comment = [
            'comment',
            'work_in_absence',
            'full_day',
            'to_notify',
        ]

        if gap_diff:
            gap_diff = {key: value for key, value in gap_diff.items() if key not in exclude_from_comment}
            if not gap_diff:
                logger.info('Skipping comment for gap %s - no gap diff', self.gap['id'])
                return

        issue_comment = self._get_issue_comment(tag, gap_diff)
        if not issue_comment:
            logger.info('Skipping comment for gap %s - no comment constructed', self.gap['id'])
            return

        author_login = None
        if self.modifier['id'] == self.gap['created_by_id'] or self.modifier['id'] == self.person['id']:
            author_login = get_user_by_uid(self.modifier['uid']).username

        gap_id = self.gap['id']
        reset_dates_params = None

        if reset_dates and gap_diff and ('date_from' in gap_diff or 'date_to' in gap_diff):
            date_to = self.gap['date_to']
            if self.gap['full_day']:
                date_to = date_to - timedelta(days=1)

            reset_dates_params = self._get_update_issue_params_for_reset_dates(date_to)

        GapTasks.schedule_ordered_task(
            gap_id,
            callable_or_task=comment_issue_for_gap,
            kwargs={
                'gap_id': gap_id,
                'issue_key': issue_key,
                'issue_comment': issue_comment,
                'tag': tag,
                'author_login': author_login,
                'login_to_call': login_to_call,
                'reset_dates_params': reset_dates_params,
                'forbidden_statuses': forbidden_statuses,
            },
        )

    def _get_update_issue_params_for_reset_dates(self, date_to) -> Dict[str, Any]:
        return {
            'start': self.gap['date_from'].date().isoformat(),
            'end': date_to.date().isoformat(),
            'ignore_version_change': True,
        }
