import hashlib
import logging
from datetime import date, datetime, timedelta

from django.conf import settings

from staff.lib.utils.staff7344 import (
    is_volozh_kostyl,
    delegate_volozh_for_gap,
)

from staff.gap.controllers.utils import get_chief
from staff.gap.workflows import (
    LOCAL_WORKFLOWS,
    WORKFLOWS_RU,
    ALL_WORKFLOWS_ORDER,
)
from staff.gap.holidays import get_holidays_by_year, OutModes

logger = logging.getLogger('staff.gap.workflows.utils')


def find_workflow(workflow):
    from staff.gap.workflows.registry import library
    from staff.gap.workflows import WORKFLOWS_REGISTRY
    return library[WORKFLOWS_REGISTRY[workflow]]


def find_form(workflow):
    from staff.gap.workflows.registry import library
    from staff.gap.workflows import FORMS_REGISTRY

    return library[FORMS_REGISTRY[workflow]]


def find_workflows_by_lang(lang):
    return (
        find_workflow(name) for name in LOCAL_WORKFLOWS.get(lang, WORKFLOWS_RU)
    )


def workflows_for_ui():
    _workflows = []
    for name in ALL_WORKFLOWS_ORDER:
        if name == 'duty':
            continue
        workflow = find_workflow(name)
        _workflows.append({
            'key': name,
            'ui_key': workflow.ui_key,
        })
    return _workflows


def edit_days_reached(date_to):
    return (datetime.now() - date_to).days >= 3


def confirm_by_chief_hash_action(person_login, person_id, modifier_id, modifier_id_for_action=None):
    def _modifier_id_for_action():
        chief = get_chief(person_login, ['id', 'login'])

        if not chief:
            return person_id

        if is_volozh_kostyl(chief['login'], person_login):
            chief = delegate_volozh_for_gap()

        return chief['id']

    to_hash = 'confirm_by_chief {}{}'.format(modifier_id, datetime.utcnow().microsecond)
    confirm_hash = hashlib.sha256(to_hash.encode('utf-8')).hexdigest()

    return {
        confirm_hash: {
            'modifier_id': modifier_id_for_action or _modifier_id_for_action(),
            'haction_name': 'confirm_by_chief',
        }
    }


def gap_year_changed(updated: dict) -> bool:
    if 'date_from' not in updated:
        return False
    old_year_from = updated['date_from']['old'].year
    new_year_from = updated['date_from']['new'].year
    return old_year_from != new_year_from


def get_min_mandatory_vacation_date_to(date_from: date, geo_id: int) -> date:
    holidays = get_holidays_by_year(date_from.year, geo_id, OutModes.holidays)
    duration = settings.MANDATORY_VACATION_DURATION

    working_days_counter = 0
    offset = 0
    while working_days_counter < duration:
        if date_from + timedelta(days=offset) not in holidays:
            working_days_counter += 1
        offset += 1

    return date_from + timedelta(days=offset)


def get_date_n_business_days_before(start_date: date, n: int, geo_id: int) -> date:
    working_days = get_holidays_by_year(year=start_date.year, geo_id=geo_id, out_mode=OutModes.weekdays)

    working_days_counter = 0
    offset = 0
    while working_days_counter < n:
        prev_day = start_date - timedelta(days=offset)

        if prev_day.year != start_date.year:
            break

        if prev_day in working_days:
            working_days_counter += 1
        offset += 1

    return start_date - timedelta(days=offset)
