from datetime import date, datetime, time, timedelta
from typing import Dict, Any

from django.conf import settings

from staff.lib.utils.ordered_choices import OrderedChoices
from staff.map.models import COUNTRY_CODES
from staff.person.models import Staff
from staff.person.tasks import FULL_YANDEX

from staff.gap.controllers.utils import get_chief, get_full_vacation_days
from staff.gap.exceptions import (
    MandatoryVacationError,
    MandatoryVacationYearChangedError,
    MandatoryVacationTooShortError,
    MandatoryVacationWithHolidaysTooShortError,
    WrongWorkflowStateError,
)
from staff.gap.workflows.base_workflow import BaseWorkflow
from staff.gap.workflows.decorators import allowed_states
from staff.gap.workflows.choices import GAP_STATES as GS
from staff.gap.workflows.utils import (
    confirm_by_chief_hash_action,
    gap_year_changed,
    get_min_mandatory_vacation_date_to,
    get_date_n_business_days_before,
)

GAP_TYPES = OrderedChoices(
    ('CREATE', 'create'),
    ('CANCEL', 'cancel'),
    ('MODIFY', 'modify'),
)


class VacationWorkflow(BaseWorkflow):

    editable_fields = [
        'is_selfpaid',
        'countries_to_visit',
    ] + BaseWorkflow.editable_fields

    workflow = 'vacation'
    ui_key = 'gap.workflow.vacation'
    gap_type = 'vacation'
    verbose_name = 'Отпуск'
    verbose_name_en = 'Vacation'
    color = '#73c9eb'

    template_context = {
        'new_gap_head_ru': 'Отпуск создан',
        'new_gap_head_en': 'Vacation was created',
        'edit_gap_head_ru': 'Отпуск изменён',
        'edit_gap_head_en': 'Vacation was edited',
        'cancel_gap_head_ru': 'Отпуск отменён',
        'cancel_gap_head_en': 'Vacation was canceled',
    }

    def sync_for_belarus(self, gap_type, data=None):
        person = Staff.objects.get(login=self.gap['person_login'])

        if person.affiliation not in FULL_YANDEX:
            return
        if person.office.get_country_code() != COUNTRY_CODES.BELARUS:
            return
        if person.vacation is None:
            return

        if gap_type == GAP_TYPES.CREATE:
            # TODO: is_selfpaid???
            person.vacation -= get_full_vacation_days(data['date_from'], data['date_to'])

        elif gap_type == GAP_TYPES.CANCEL:
            if not self.gap.get('is_selfpaid', False):
                person.vacation += get_full_vacation_days(self.gap['date_from'], self.gap['date_to'])

        elif gap_type == GAP_TYPES.MODIFY:
            if 'is_selfpaid' in data:
                if data['is_selfpaid']['old']:
                    person.vacation -= get_full_vacation_days(self.gap['date_from'], self.gap['date_to'])
                else:
                    if 'date_from' in data:
                        person.vacation += get_full_vacation_days(data['date_from']['old'], data['date_to']['old'])
                    else:
                        person.vacation += get_full_vacation_days(self.gap['date_from'], self.gap['date_to'])
            else:
                if not self.gap.get('is_selfpaid', False):
                    if 'date_from' in data:
                        person.vacation += get_full_vacation_days(data['date_from']['old'], data['date_to']['old'])
                        person.vacation -= get_full_vacation_days(self.gap['date_from'], self.gap['date_to'])

        person.save()

    def new_gap(self, data):

        self._create_self_gap(data)

        self._new_gap_email(approver=get_chief(self.gap['person_login']))

        if data.get('is_selfpaid', False):
            tag = 'new_selfpaid_vacation'
        else:
            tag = 'new_vacation'

            self.sync_for_belarus(GAP_TYPES.CREATE, data)

        return self.new_issue(tag)

    def prepare_issue_params(self, tag: str, append_followers=None, issue_unique_prefix=None) -> Dict[str, Any]:
        params = super(VacationWorkflow, self).prepare_issue_params(
            tag,
            append_followers=append_followers,
            issue_unique_prefix=None,
        )
        if not params:
            return params

        params['staffDate'] = params['end']
        return params

    def get_initial_issue_params(self):
        result = {
            'legalEntity2': self.person['organization__st_translation_id'],
            'currentHead': self.chief['login'] if self.chief else None,
            'hr': self.hrpb_logins[0] if self.hrpb_logins else None,
        }

        if self.gap.get('mandatory'):
            result[settings.STARTREK_MANDATORY_GAP_FIELD] = 'Y'

        return result

    def _append_gap(self, data):
        return {
            'is_selfpaid': data.get('is_selfpaid', False),
            'hactions': confirm_by_chief_hash_action(
                self.person['login'],
                self.person['id'],
                self.modifier['id'],
            ),
            'countries_to_visit': data.get('countries_to_visit', []),
            'mandatory': data.get('mandatory', False),
            'vacation_updated': data.get('vacation_updated', False),
            'deadline': data.get('deadline', None),
            'geo_id': data.get('geo_id', 0),
            'notifications_sent': data.get('reminders_sent', []),
        }

    def _append_base_default(self):
        return {
            'is_selfpaid': False,
            'hactions': confirm_by_chief_hash_action(
                self.person['login'],
                self.person['id'],
                self.modifier['id'],
            ),
            'countries_to_visit': [],
            'mandatory': False,
            'vacation_updated': False,
            'deadline': None,
            'geo_id': 0,
            'reminders_sent': [],
        }

    @allowed_states([GS.NEW])
    def confirm_gap(self):
        self._set_state(GS.CONFIRMED)
        self.gap['confirmed_by_id'] = self.modifier['id']

        self._update()
        self.issue_comment('confirm_vacation', forbidden_statuses=['closed', 'cancelled'])

    @allowed_states([GS.CONFIRMED])
    def sign_gap(self):
        self._set_state(GS.SIGNED)

        self._update()
        self.issue_comment('sign_vacation')

    @allowed_states([GS.NEW, GS.CONFIRMED, GS.SIGNED])
    def edit_gap(self, data):
        updated = self._update_editable(data, self.editable_fields)
        if not updated:
            return

        # считаем обязательный отпуск обновленным, если изменилось что-то кроме
        # согласования
        self.gap['vacation_updated'] = len(updated) > 1 or 'state' not in updated
        self.sync_for_belarus(GAP_TYPES.MODIFY, updated)

        modifier_user = Staff.objects.get(id=self._modifier_id).user

        unconfirm_issue = False
        if 'date_from' in updated:
            self.gap['reminders_sent'] = []

            if (
                self.gap.get('mandatory', False)
                and not modifier_user.has_perm('gap.can_edit_mandatory_vacation')
            ):
                self._check_mandatory_vacation_conditions(updated)

            if self.must_be_unconfirmed(data):
                self.unconfirm()
                unconfirm_issue = True
            self.gap['last_day_when_commented_vacation_approval_reminder'] = None

        self._update()
        self._new_to_notify(updated)

        if not updated:
            return

        if unconfirm_issue:
            self.issue_comment('edit_gap_unconfirm', gap_diff=updated, reset_dates=True)
        else:
            self.issue_comment('edit_gap', gap_diff=updated, reset_dates=True)

        self._edit_gap_email(updated, approver=get_chief(self.gap['person_login']))

    def _check_mandatory_vacation_conditions(self, gap_diff: dict) -> None:
        if gap_year_changed(gap_diff):
            raise MandatoryVacationYearChangedError()

        vacation_duration = gap_diff['date_to']['new'] - gap_diff['date_from']['new']

        if vacation_duration < timedelta(settings.MANDATORY_VACATION_DURATION):
            min_date_to = gap_diff['date_from']['new'] + timedelta(
                settings.MANDATORY_VACATION_DURATION)
            raise MandatoryVacationTooShortError(min_date_to.date() - timedelta(1))

        try:
            person = Staff.objects.get(login=self.gap['person_login'])
            person_geo_id = person.office.city.geo_id
        except KeyError:
            raise MandatoryVacationError(
                'no person_login given',
                'no-person-login',
            )
        except Staff.DoesNotExist:
            raise MandatoryVacationError(
                'invalid person_login %s' % self.gap['person_login'],
                'invalid-person-login',
            )

        min_date_to = get_min_mandatory_vacation_date_to(
            gap_diff['date_from']['new'].date(),
            person_geo_id,
        )

        if min_date_to > gap_diff['date_to']['new'].date():
            raise MandatoryVacationWithHolidaysTooShortError(min_date_to - timedelta(1))

        new_deadline = get_date_n_business_days_before(
            start_date=gap_diff['date_from']['new'].date(),
            n=settings.MANDATORY_VACATION_DEADLINE_DAYS,
            geo_id=person_geo_id,
        )

        if new_deadline <= date.today():
            raise MandatoryVacationError(
                'Deadline already passed',
                'Непрерывный отпуск можно изменить не позднее чем за 10 рабочих дней до его начала',
            )

        self.gap['deadline'] = datetime.combine(
            new_deadline,
            time.min,
        )

    def must_be_unconfirmed(self, data: dict) -> bool:
        return (
            self.gap['state'] in [GS.CONFIRMED, GS.SIGNED]
            and not (len(data) == 1 and 'reminders_sent' in data)
        )

    def unconfirm(self):
        self._set_state(GS.NEW)
        self.gap['confirmed_by_id'] = None
        self.gap['commented_vacation_statement_reminder'] = False
        self._clean_haction('confirm_by_chief')
        self.gap['hactions'].update(confirm_by_chief_hash_action(
            self.person['login'],
            self.person['id'],
            self.modifier['id'],
        ))

    @allowed_states([GS.NEW, GS.CONFIRMED, GS.SIGNED])
    def cancel_gap(self, send_email=True, issue_comment_tag='cancel_vacation'):
        self._set_state(GS.CANCELED)
        self._update()
        self.sync_for_belarus(GAP_TYPES.CANCEL)

        if send_email:
            self._cancel_gap_email(approver=get_chief(self.gap['person_login']))

        self.issue_comment(issue_comment_tag)

    def confirm_by_chief_haction(self, haction_hash):
        assert haction_hash in self.gap['hactions']
        self.gap['hactions'].pop(haction_hash)
        try:
            self.confirm_gap()
        except WrongWorkflowStateError:
            self._update()

    def _get_update_issue_params_for_reset_dates(self, date_to) -> Dict[str, Any]:
        issue_params = super()._get_update_issue_params_for_reset_dates(date_to)
        if 'end' in issue_params:
            issue_params['staffDate'] = issue_params['end']
        return issue_params
