from django.contrib.auth.models import AnonymousUser, Permission
from django.db import models

from staff.lib.exceptions import SearchError
from staff.lib.models.base import IntranetMpttModel, IntranetManager, AtomicSaveModel
from staff.lib.pg_lock import take_pg_lock
from staff.lib.utils.ordered_choices import OrderedChoices
from staff.person.models import Staff
from staff.users.models import User

GROUP_TYPE_CHOICES = OrderedChoices(
    ('WIKI', 0, 'wiki'),
    ('DEPARTMENT', 1, 'department'),
    ('SERVICE', 2, 'service'),
    ('SERVICEROLE', 3, 'servicerole'),
)

SERVICEROLE_GROUP_TYPES = (
    ('adm', 'Админы (%s)'),
    ('dev', 'Разработчики (%s)'),
    ('pm', 'Менеджеры (%s)'),
    ('test', 'Тестирование (%s)'),
    ('support', 'Саппорт (%s)'),
    ('users', 'Другие (%s)'),
)


class GroupNotFound(SearchError):
    code = 'no_such'


class GroupFoundMultiple(SearchError):
    code = 'many'


class GroupManager(IntranetManager):
    intranet_root_kwargs = {'url': '__wiki__', 'level': 0}

    def get_intranet_root(self):
        return self.get(**self.intranet_root_kwargs)

    def get_admins(self):
        return self.get(name='Admins', parent__url='__wiki__', parent__level=0)

    departments_root_kwargs = {'url': '__departments__', 'level': 0}

    def get_departments_root(self):
        return self.get(**self.departments_root_kwargs)

    services_root_kwargs = {'url': '__services__', 'level': 0}

    def get_services_root(self):
        return self.get(**self.services_root_kwargs)

    intra_services_root_kwargs = {'url': '__intranet_services__', 'level': 0}

    def get_intra_services_root(self):
        return self.get(**self.intra_services_root_kwargs)

    def is_admin(self, user):
        if not user or isinstance(user, AnonymousUser) or not Staff:
            return False
        kwargs = {'group__name': 'Admins', 'group__parent__url': '__wiki__', 'group__parent__level': 0}
        if isinstance(user, User):
            if not user.is_authenticated():
                return False
            kwargs.update({'staff__login': user.username})
        elif isinstance(user, str):
            kwargs.update({'staff__login': user})
        elif isinstance(user, Staff):
            kwargs.update({'staff': user})
        return GroupMembership.objects.filter(**kwargs).exists()

    def get_type_queryset(self, type_):
        if type_ == 'intra':
            type_ = 'wiki'

        available = [key for __, key in GROUP_TYPE_CHOICES.choices()]

        if type_ not in available:
            raise ValueError('Type must be one of: %s' % ', '.join(available))

        return self.get_query_set().filter(type=GROUP_TYPE_CHOICES.get_key(type_))

    def search_exactly(self, query, typ):
        """
        Search exact fit group by query and type
        Used in interfaces as defense of groups autocomlete from incorrect input
        """
        qs = self.get_type_queryset(typ).filter(intranet_status__gt=0)

        try:
            try:
                r = qs.get(name=query)
            except Group.DoesNotExist:
                try:
                    r = qs.get(url=query)
                except Group.DoesNotExist:
                    raise GroupNotFound('Group "%s" has not been found' % query)
        except Group.MultipleObjectsReturned:
            raise GroupFoundMultiple('There are several groups named "%s"' % query)
        return r


class Group(IntranetMpttModel):
    class MPTTMeta:
        order_insertion_by = ['position']

    class Meta(IntranetMpttModel.Meta):
        db_table = 'intranet_group'
        unique_together = [('parent', 'role_scope_id')]
        index_together = [
            ('tree_id', 'lft', 'rght'),
        ]

    objects = GroupManager()

    OPTION = 808  # evil hidden group
    department = models.OneToOneField('Department', null=True, blank=True)

    service_id = models.PositiveIntegerField(null=True, blank=True, db_index=True)
    role_scope_id = models.CharField(max_length=127, null=True, blank=True)

    parent = models.ForeignKey('self', null=True, blank=True, related_name='children')
    name = models.CharField(max_length=1270, db_index=True)
    name_en = models.CharField(max_length=1270, null=True, db_index=True)
    url = models.CharField(max_length=255, unique=True, db_index=True)
    code = models.CharField(max_length=127, db_index=True, default='')
    description = models.TextField()
    position = models.IntegerField(null=True, default=0)

    type = models.PositiveSmallIntegerField(
        choices=GROUP_TYPE_CHOICES.choices(),
        default=GROUP_TYPE_CHOICES.WIKI,
        blank=False,
    )

    members = models.ManyToManyField('Staff', through='GroupMembership', related_name='in_groups')
    responsible = models.ManyToManyField(Staff, through='GroupResponsible', related_name='responsible_for_groups')
    permissions = models.ManyToManyField(Permission, related_name='groups', blank=True)

    # денормализация: число внешних сотрудников в группе.
    externals_count = models.PositiveIntegerField(default=0)
    yandex_count = models.PositiveIntegerField(default=0)
    yamoney_count = models.PositiveIntegerField(default=0)

    service_tags = models.CharField(max_length=128, default='', null=False, blank=True)
    parent_service_id = models.IntegerField(null=True, blank=True, default=None)  # id родительского сервиса

    def get_all_members(self, include_dismissed=False):
        """
        Return all members of this group and all its subgroups as
        QuerySet object. Dismissed employees are ignored.
        """

        members = Staff.objects.filter(
            groupmembership__group__tree_id=self.tree_id,
            groupmembership__group__lft__gte=self.lft,
            groupmembership__group__rght__lte=self.rght
        )

        if not include_dismissed:
            members = members.filter(is_dismissed=False)

        return members

    @property
    def all_members(self):
        """
        Property stands for backward compatibility. New code should use
        `get_all_members` method that returns QuerySet instead of list
        and thus can be further tuned
        """
        return list(self.get_all_members())

    @property
    def is_intranet_service(self):
        url = Group.objects.intra_services_root_kwargs['url']
        ancs = self.get_ancestors()
        return self.url == url or (len(ancs) and ancs[0].url == url)

    @property
    def is_intranet(self):
        return self.is_wiki

    @property
    def is_wiki(self):
        return self.type == GROUP_TYPE_CHOICES.WIKI

    @property
    def is_service(self):
        return self.type == GROUP_TYPE_CHOICES.SERVICE

    @property
    def is_servicerole(self):
        return self.type == GROUP_TYPE_CHOICES.SERVICEROLE

    @property
    def is_department(self):
        return self.type == GROUP_TYPE_CHOICES.DEPARTMENT

    def is_responsible(self, user):
        if not user or isinstance(user, AnonymousUser):
            return False

        kw = {}
        if not isinstance(user, User):
            kw['staff'] = user
        else:
            if not user.is_authenticated():
                return False
            kw['staff__login'] = user.username
        kw['staff__is_dismissed'] = False
        kw['group__in'] = list(self.get_ancestors()) + [self]

        if GroupResponsible.objects.filter(**kw).count():
            return True

        if Group.objects.is_admin(user):
            return True

        return False

    def __str__(self):
        return self.name

    def save(self, *args, **kwargs):
        if self.is_department:
            five_minutes = 60 * 5 * 1000
            take_pg_lock('department_group_lock', five_minutes, five_minutes)

        super().save(*args, **kwargs)


class GroupMembership(AtomicSaveModel):
    staff = models.ForeignKey(Staff)
    group = models.ForeignKey(Group)
    joined_at = models.DateTimeField(auto_now_add=True)

    def __str__(self):
        return str(self.staff) + ' in ' + str(self.group)

    class Meta:
        unique_together = (('staff', 'group'),)
        db_table = 'intranet_groupmembership'
        app_label = 'django_intranet_stuff'


class GroupResponsible(AtomicSaveModel):
    staff = models.ForeignKey(Staff)
    group = models.ForeignKey(Group)
    permitted_at = models.DateTimeField(auto_now_add=True)

    def __str__(self):
        return str(self.staff) + ' can manage membership of ' + str(self.group)

    class Meta:
        unique_together = (('staff', 'group'),)
        db_table = 'intranet_groupresponsible'
        app_label = 'django_intranet_stuff'


class AutoGroupTemplate(models.Model):
    """Шаблон для автогенерируемых групп."""
    group = models.OneToOneField(
        verbose_name='группа',
        to='django_intranet_stuff.Group',
        limit_choices_to={'type': GROUP_TYPE_CHOICES.WIKI, 'level__gte': 1},
        null=True,
        blank=True,
        help_text='Если не выбрано -- новая группа будет создана',
    )
    name = models.CharField(
        verbose_name='Имя группы',
        unique=True,
        max_length=127,
        help_text='Будет использовано при создании группы по шаблону',
    )
    code = models.SlugField(
        verbose_name='Код группы',
        unique=True,
        max_length=127,
        help_text='Будет последней частью url созданной группы.',
    )
    condition = models.TextField(
        verbose_name='Условия для фильтрации',
        help_text='Пример: [{"gender": "M", "is_dismissed": false}]',
        blank=True,
    )
    exclusion = models.TextField(
        verbose_name='Условия для исключения',
        blank=True,
    )


class ServiceUpdateTasksQueue(models.Model):
    fake_identity = models.CharField(max_length=14)
    service_id = models.IntegerField(null=False, db_index=True)
    args = models.TextField()
    callable = models.CharField(max_length=128)
    module = models.CharField(max_length=512)
    fail_count = models.IntegerField(default=0, null=False)
