from datetime import datetime
import logging
from uuid import uuid4

from staff.person.models import Staff, AFFILIATION
from staff.groups.models import Group, GroupMembership, GROUP_TYPE_CHOICES

from ..models import AutoGroupTemplate
from ..signals import group_deleted, group_updated, group_created
from ..signals import group_member_added, group_member_deleted


logger = logging.getLogger(__name__)


class GroupCtl(object):
    """
    ВАЖНО: Контроллер группы не должен никогда отдавать состав
    группы Group.OPTION.
    """

    UPDATABLE_FIELDS = [
        'name',
        'name_en',
        'code',
        'description',
        'position',
        'externals_count',
        'yandex_count',
        'yamoney_count',
        'service_id',
        'role_scope_id',
        'parent',
    ]

    type = None

    def __init__(self, group=None, request=None):
        super(GroupCtl, self).__init__()
        self.group = group
        self.request = request

    def create(self, parent=None, commit=False):
        random_url = 'new_' + uuid4().hex[:8]
        if parent is None:
            native_lang = ''
        else:
            native_lang = parent.native_lang

        # set right type
        self.group = Group(
            parent=parent,
            type=self.type,
            url=random_url,
            intranet_status=0,
            native_lang=native_lang,
            created_at=datetime.now(),
            modified_at=datetime.now(),
        )

        if commit:
            self.group.save(force_insert=True)

        self._send_signal('group_created')
        return self.group

    def update(self, data: dict, enable: bool = True):
        updated_fields = set()

        for field, value in data.items():
            if self.UPDATABLE_FIELDS:
                setattr(self.group, field, value)
                updated_fields.add(field)

        if enable and self.group.intranet_status == 0:
            logging.info('Group resurrection %s', self.group.id)

            self.group.intranet_status = 1
            updated_fields.add('intranet_status')

            if 'code' in data or 'parent' in data:
                self.group.url = self.url
                updated_fields.add('url')

        if self.group.id is None:
            self.group.save()
        else:
            self.group.save(update_fields=updated_fields)
        self._send_signal('group_updated')

        return self.group

    def create_filled(self, data, parent=None):
        # TODO: возможно нужно переделать этот метод и create
        self.create(parent=parent)
        return self.update(data)

    def delete(self):
        if self.group.intranet_status != 0:
            random_url = 'del_' + uuid4().hex[:8]
            self.group.intranet_status = 0
            self.group.url = random_url
            self.group.save(force_update=True)
            self._send_signal('group_deleted')

    def get_members(self):
        """Возвращает сотрудников, непосредственно состоящих в группе.
         Метод НЕ возвращает сотрудников, принадлежащих подгруппам этой группы.
         """
        if self.group.id == Group.OPTION:
            return Staff.objects.none()
        return self.group.members.active().order_by('-join_at')

    def get_external_members(self):
        """Возвращает внешних сотрудников, непосредственно состоящих в группе.
        Метод НЕ возвращает сотрудников, принадлежащих подгруппам этой группы.
        """
        return self.get_members().filter(affiliation=AFFILIATION.EXTERNAL)

    def get_yandex_members(self):
        """Возвращает сотрудников Яндекс, непосредственно состоящих в группе.
        Метод НЕ возвращает сотрудников, принадлежащих подгруппам этой группы.
        """
        return self.get_members().filter(affiliation=AFFILIATION.YANDEX)

    def get_yamoney_members(self):
        """Возвращает сотрудников Яндекс.Денег, непосредственно состоящих в группе.
        Метод НЕ возвращает сотрудников, принадлежащих подгруппам этой группы.
        """
        return self.get_members().filter(affiliation=AFFILIATION.YAMONEY)

    def get_all_members(self):
        """Возвращает сотрудников в группе и рекурсивно сотрудников из
        подгрупп этой группы. Включая уволенных сотрудников.
        Повторов быть не должно.
        """
        if self.group.id == Group.OPTION:
            return Staff.objects.none()
        return Staff.objects.filter(
            groupmembership__group__tree_id=self.group.tree_id,
            groupmembership__group__lft__gte=self.group.lft,
            groupmembership__group__rght__lte=self.group.rght,
        ).distinct()

    def get_all_external_members(self):
        """Возвращает внешних сотрудников, состоящих в группе и рекурсивно
        внешних сотрудников из подгрупп этой группы.
        Повторов быть не должно.
        """
        return self.get_all_members().filter(affiliation=AFFILIATION.EXTERNAL)

    def get_all_yandex_members(self):
        """Возвращает сотрудников Яндекса, состоящих в группе и рекурсивно
        внешних сотрудников из подгрупп этой группы.
        Повторов быть не должно.
        """
        return self.get_all_members().filter(affiliation=AFFILIATION.YANDEX)

    def get_all_yamoney_members(self):
        """Возвращает сотрудников Яндекс.Денег, состоящих в группе и рекурсивно
        внешних сотрудников из подгрупп этой группы.
        Повторов быть не должно.
        """
        return self.get_all_members().filter(affiliation=AFFILIATION.YAMONEY)

    def get_members_count(self):
        """Число сотрудников непосредственно в группе."""
        return self.get_members().count()

    def get_all_members_count(self):
        """Число неуволенных сотрудников в группе и ее подгруппах."""
        return self.get_all_members().filter(is_dismissed=False).count()

    def get_externals_count(self):
        """Число неуволенных внешних сотрудников в группе."""
        return self.get_external_members().filter(is_dismissed=False).count()

    def get_all_externals_count(self):
        """Число неуволенных внешних сотрудников в группе и ее подгруппах."""
        return self.get_all_external_members().filter(is_dismissed=False).count()

    def get_yandex_members_count(self):
        """Число неуволенных сотрудников Яндекс в группе."""
        return self.get_yandex_members().filter(is_dismissed=False).count()

    def get_all_yandex_members_count(self):
        """Число неуволенных сотрудников Яндекс в группе и ее подгруппах."""
        return self.get_all_yandex_members().filter(is_dismissed=False).count()

    def get_yamoney_members_count(self):
        """Число неуволенных сотрудников Яндекс.Деньги в группе."""
        return self.get_yamoney_members().filter(is_dismissed=False).count()

    def get_all_yamoney_members_count(self):
        """Число неуволенных сотрудников Яндекс.Деньги в группе и ее подгруппах."""
        return self.get_all_yamoney_members().filter(is_dismissed=False).count()

    def add_member(self, new_member):
        membership, created = GroupMembership.objects.get_or_create(
            staff=new_member, group=self.group,
        )
        if created:
            self._send_signal('group_member_added', staff=new_member)
        return created

    def add_members_by_logins(self, logins):
        new_members = Staff.objects.filter(login__in=logins)

        members_added_count = 0
        for member in new_members:
            created = self.add_member(member)
            if created:
                members_added_count += 1

        return members_added_count

    def delete_member(self, member):
        membership = GroupMembership.objects.get(
            staff=member, group=self.group,
        )
        membership.delete()
        self._send_signal('group_member_added', staff=member)

    def delete_all_members(self):
        for member in self.get_members():
            self.delete_member(member)

    def is_empty(self):
        return not self.has_descendants() and not self.has_members()

    def has_descendants(self):
        descendants = self.base_queryset.filter(
            lft__gt=self.group.lft,
            rght__lt=self.group.rght
        )
        return descendants.exists()

    def has_members(self):
        return GroupMembership.objects.filter(group=self.group).exists()

    def has_member(self, staff):
        return self.get_members().filter(id=staff.id).exists()

    @property
    def id(self):
        return self.group.id

    @property
    def base_queryset(self):
        return Group.objects.active()

    @property
    def url(self):
        """Для корневых групп url равен code, напр. '__wiki__'.
        Для групп первого уровня url также равен code, т.е. он не должен
        включать префикс типа __wiki__.
        Все остальные формируют урл из родительского + свой code.
        """
        if self.level in (0, 1):
            return self.code
        else:
            return self.parent.url + '_' + self.code

    @property
    def code(self):
        return self.group.code

    @property
    def level(self):
        if self.group.level is not None:
            return self.group.level
        elif self.parent is not None:
            return self.parent.level + 1
        else:
            return 0

    @property
    def parent(self):
        return self.group.parent

    @property
    def is_autogroup(self):
        if self.type != GROUP_TYPE_CHOICES.WIKI:
            return False
        return AutoGroupTemplate.objects.filter(group_id=self.id).exists()

    def _send_signal(self, action, staff=None):
        if self.request is None:
            # логируем ли мы в аудит дергания из менеджмент-команд?
            return
        signal = {
            'created': group_created,
            'updated': group_updated,
            'deleted': group_deleted,
            'member_added': group_member_added,
            'member_deleted': group_member_deleted,
        }[action]
        signal.send(
            sender=self.__class__,
            request=self.request,
            group=self.group,
            staff=staff,
        )

    def dehydrate(self):
        return {
            'id': self.group.id,
            'name': self.group.name,
            'url': self.group.url,
            'description': self.group.description,
            'ancestors': list(
                self.group.get_ancestors().values('name', 'url')
            ),
        }


class WikiGroupCtl(GroupCtl):

    type = GROUP_TYPE_CHOICES.WIKI

    @property
    def url(self):
        # TODO: проставить код для групп нулевого уровня?
        if self.level == 0:
            return self.code
        elif self.level == 1:
            return 'category' + '_' + self.code
        elif self.level == 2:
            return self.code
        else:
            return self.parent.url + '_' + self.code

    def dehydrate(self):
        dehydrated = super(WikiGroupCtl, self).dehydrate()
        tags = dehydrated.setdefault('tags', [])
        if self.is_autogroup:
            tags.append('isAutogroup')
        if self.group.externals_count:
            tags.append('hasExternals')
        if not self.get_members_count():
            tags.append('hasNoMembers')
        return dehydrated


class DepartmentGroupCtl(GroupCtl):

    type = GROUP_TYPE_CHOICES.DEPARTMENT

    def dehydrate(self):
        dehydrated = super(DepartmentGroupCtl, self).dehydrate()
        dehydrated['department'] = {
            'name': self.group.department and self.group.department.i_name,
            'url': self.group.department and self.group.department.url,
        }
        return dehydrated


class ServiceGroupCtl(GroupCtl):

    UPDATABLE_FIELDS = [
        'native_lang',
        'name',
        'name_en',
        'type',
        'code',
        'description',
        'position',
        'externals_count',
        'yandex_count',
        'yamoney_count',
        'service_id',
        'role_scope_id',
        'parent',
        'service_tags',
        'parent_service_id',
    ]

    type = GROUP_TYPE_CHOICES.SERVICE

    def dehydrate(self):
        dehydrated = super(ServiceGroupCtl, self).dehydrate()
        dehydrated['service'] = {
            'id': self.group.service_id,
            'name': self.group.service_id and self.group.name,
        }
        return dehydrated

    @property
    def url(self):
        if self.parent and self.parent.code == '__services__':
            return 'svc_' + self.code
        else:
            return super(ServiceGroupCtl, self).url


class ServiceRoleGroupCtl(GroupCtl):

    UPDATABLE_FIELDS = [
        'name',
        'type',
        'code',
        'description',
        'position',
        'externals_count',
        'yandex_count',
        'yamoney_count',
        'service_id',
        'role_scope_id',
        'parent',
        'service_tags',
        'parent_service_id',
    ]

    type = GROUP_TYPE_CHOICES.SERVICEROLE

    @property
    def url(self):
        return 'role_' + super().url
