from datetime import datetime

from django.core.exceptions import (
    PermissionDenied,
    ObjectDoesNotExist,
)
from django.db.models import Q

from staff.groups.models import Group, AutoGroupTemplate, GROUP_TYPE_CHOICES
from staff.groups.permissions import can_add_group, can_edit_group, can_delete_group

WIKI_ROOT_ID = 1
STATUS_ACTIVE = 1
STATUS_INACTIVE = 0


def get_wiki_list(**filter_by):
    qs = (
        Group.objects
        .filter(
            type=GROUP_TYPE_CHOICES.WIKI,
            intranet_status=STATUS_ACTIVE,
            autogrouptemplate__isnull=True,
        )
        .exclude(id=WIKI_ROOT_ID)
        .order_by('name')
    )

    if filter_by:
        or_q = Q()
        for field, value in filter_by.items():
            or_q |= Q(**{field: value})
        qs = qs.filter(or_q)

    return qs


def deactivate_group(user, group_id):
    group = check_and_get_wiki_group(group_id)

    if not can_delete_group(user):
        raise PermissionDenied()

    group.intranet_status = STATUS_INACTIVE
    group.modified_at = datetime.now()
    group.save()


def save_wiki_group(user, form_data, group_id):
    now = datetime.now()

    if group_id is not None:
        group = check_and_get_wiki_group(group_id)

        if not can_edit_group(group, user):
            raise PermissionDenied()

    else:
        if not can_add_group(user):
            raise PermissionDenied()

        group = Group()
        group.type = GROUP_TYPE_CHOICES.WIKI
        group.code = form_data['code']
        group.parent = Group.objects.get(id=WIKI_ROOT_ID)
        group.created_at = now
        group.url = group.code.lower()

    group.name = form_data['name'].strip()
    group.description = form_data['description'].strip()
    group.modified_at = now

    group.save()


def check_and_get_wiki_group(group_id):
    try:
        group = Group.objects.get(id=group_id)
    except Group.DoesNotExist as e:
        raise ObjectDoesNotExist(e)

    if (group.type != GROUP_TYPE_CHOICES.WIKI or
            group_id == WIKI_ROOT_ID or
            group.intranet_status != STATUS_ACTIVE or
            AutoGroupTemplate.objects.filter(group_id=group_id).exists()):
        raise PermissionDenied()

    return group
