import logging

from django.db.models import Q

from staff.person.models import Staff
from staff.groups.models import Group, GroupMembership, GROUP_TYPE_CHOICES

from staff.groups.service.commands import (
    CreateMember,
    DeleteMember,
    CreateServiceGroup,
    UpdateServiceGroup,
    DeleteServiceGroup,
    CreateServiceRoleGroup,
    UpdateServiceRoleGroup,
    DeleteServiceRoleGroup,
)
from staff.groups.service.constants import TERMINATED_SERVICE_STATES
from staff.groups.service.datasource import get_service_members
from staff.groups.service.helpers import normalize_data


logger = logging.getLogger(__name__)


class ServiceDataProcessor(object):

    @classmethod
    def process(cls, data):
        deleted_groups = set()

        for service_id, details in data:

            state = details.get('state')

            if state in TERMINATED_SERVICE_STATES:
                deleted_groups.add(service_id)

            try:
                group = Group.objects.get(service_id=service_id)
            except Group.DoesNotExist:
                yield CreateServiceGroup(service_id, details)
            else:
                yield UpdateServiceGroup(group, service_id, details)

            for cmd in cls._process_members(service_id, details):
                yield cmd

        yield from cls._delete_groups(deleted_groups)

    @classmethod
    def _process_members(cls, service_id, data):
        service_group = Group.objects.get(service_id=service_id, type=GROUP_TYPE_CHOICES.SERVICE)

        exist_subgroups = dict(
            (g.role_scope_id, g)
            for g in Group.objects.filter(
                parent__service_id=service_id,
                type=GROUP_TYPE_CHOICES.SERVICEROLE,
            )
        )

        existing_members = {}

        raw_team = get_service_members(
            lookup={
                'service__with_descendants': service_id,
                'use_inheritance_settings': True,
                'is_exportable': True,
                'fields': 'person.login,role.scope.slug,role.scope.name',
                'page_size': 100,
            },
            timeout=(2, 5, 10),
        )
        normalized_team = (normalize_data(raw_member) for raw_member in raw_team)
        team = [member for member in normalized_team if member['role'].get('scope') is not None]
        logins = [member['person']['login'] for member in team]
        staff_map = {staff.login: staff for staff in Staff.objects.filter(login__in=logins)}

        person_groups_ids = {}
        for membership in GroupMembership.objects.filter(staff__login__in=logins).values('staff__login', 'group_id'):
            person_groups_ids.setdefault(membership['staff__login'], set()).add(membership['group_id'])

        for member in team:
            # Ролевая группа
            login = member['person']['login']
            scope = member['role']['scope']
            scope_id = scope['slug']
            person = staff_map[login]

            scope_group = exist_subgroups.get(scope_id)
            if scope_group is None:
                yield CreateServiceRoleGroup(service_group, scope)
                scope_group = Group.objects.get(parent__service_id=service_id, role_scope_id=scope_id)
                exist_subgroups[scope_id] = scope_group
            else:
                yield UpdateServiceRoleGroup(scope_group, service_group, scope, state=data['state'])

            # Членство в самой сервисной группе.
            # удалить этот блок, когда решим не добавлять людей в сервисные группы
            existing_members.setdefault(service_group.id, set()).add(person.id)
            if service_group.id not in person_groups_ids.get(login, []):
                person_groups_ids.setdefault(login, set()).add(service_group.id)
                yield CreateMember(service_id, service_group, person)

            # Членство в ролевой группе
            if scope_group:
                existing_members.setdefault(scope_group.id, set()).add(person.id)
                if scope_group.id not in person_groups_ids.get(login, []):
                    person_groups_ids.setdefault(login, set()).add(scope_group.id)
                    yield CreateMember(service_id, scope_group, person)

        yield from cls._delete_members(service_id, service_group, existing_members)

    @classmethod
    def _delete_members(cls, service_id, service_group, existing_members):

        for group_id, existing_members_set in existing_members.items():
            unnecessary_members = (
                GroupMembership.objects
                .filter(group=group_id)
                .exclude(staff__in=existing_members_set)
            )

            for groupmember in unnecessary_members:
                yield DeleteMember(service_id, groupmember)

        empty_role_members = (
            GroupMembership.objects
            .filter(group__parent=service_group)
            .exclude(group__in=existing_members.keys())
        )

        for groupmember in empty_role_members:
            yield DeleteMember(service_id, groupmember)

    @classmethod
    def _delete_groups(cls, deleted_group_ids):
        deleted_groups = (
            Group.objects
            .filter(
                type__in=[
                    GROUP_TYPE_CHOICES.SERVICE,
                    GROUP_TYPE_CHOICES.SERVICEROLE,
                ],
                intranet_status=1,
            )
            .filter(
                Q(service_id__in=deleted_group_ids)
                | Q(parent__service_id__in=deleted_group_ids)
            )
            .exclude(code='__services__')
        )

        for group in deleted_groups:
            if group.type == GROUP_TYPE_CHOICES.SERVICE:
                yield DeleteServiceGroup(group)
            else:
                yield DeleteServiceRoleGroup(group)
