from django import forms

from django.conf import settings
from django.core.exceptions import (
    PermissionDenied,
    ObjectDoesNotExist,
)
from django.core.urlresolvers import reverse
from django.core.paginator import Paginator, EmptyPage, PageNotAnInteger

from django.http import (
    Http404,
    HttpResponseForbidden,
    HttpResponseRedirect,
)

from django.shortcuts import render
from django.views.decorators.http import require_http_methods

from staff.groups.models import Group

from staff.lib.utils.paging import page_range

from staff.groups.objects import groups_ctl
from staff.groups.permissions import can_add_group, can_edit_group, can_delete_group


class GroupForm(forms.Form):
    name = forms.CharField(max_length=127)
    description = forms.CharField(max_length=10000, required=False)

    def clean_name(self):
        name = self.cleaned_data['name']
        if not name.strip():
            raise forms.ValidationError('Имя из пробелов — плохое имя, нехорошее.')

        return name


class NewGroupForm(GroupForm):
    invalid_message = (
        'Код не должен содержать пробельные символы, '
        'должен состоять только из латиницы, цифр и подчёркивания, '
        'не может начинаться на del, svc и dpt.'
    )
    code = forms.RegexField(
        max_length=127,
        regex=r'^((?!del)(?!svc)(?!dpt))[a-zA-Z\d_]+$',
        error_messages={'invalid': invalid_message}
    )

    def clean_code(self):
        code = self.cleaned_data['code']

        url = code.lower()
        if Group.objects.filter(url=url).exists():
            raise forms.ValidationError('URL по такому коду уже существует.')

        return code


GROUP_MEMBERS_PER_PAGE = 50


@require_http_methods(['GET'])
def groups_list_view(request):
    query = request.GET.get('query', '').strip()

    if query:
        filter_by = {
            'name__contains': query,
            'code__contains': query,
            'url__contains': query,
        }
    else:
        filter_by = {}

    paginator = Paginator(groups_ctl.get_wiki_list(**filter_by), 20)
    try:
        groups = paginator.page(request.GET.get('page'))
    except PageNotAnInteger:
        groups = paginator.page(1)
    except EmptyPage:
        groups = paginator.page(paginator.num_pages)

    groups_with_permissions = [
        (
            group,
            # cast to int cause this used in js generation, js doesn't understand True/False
            int(can_edit_group(group, request.user)),
            int(can_delete_group(request.user)),
        )
        for group in groups
    ]

    context = {
        'can_add_wiki_group': can_add_group(request.user),
        'groups': groups_with_permissions,
        'pages_nums': page_range(groups.number, paginator.num_pages, 10),
        'query': query,
        'upravlyator_host': settings.UPRAVLYATOR_HOST,
        'user': request.user,
    }

    return render(request, 'groups/groups_list.html', context)


@require_http_methods(['POST', 'GET'])
def group_save_view(request, group_id=None):
    if group_id is None:
        template_url = 'groups/create_group.html'
        group = None
    else:
        group_id = int(group_id)
        template_url = 'groups/edit_group.html'
        try:
            group = Group.objects.get(id=group_id, intranet_status=1)
        except Group.DoesNotExist:
            raise Http404

    if request.method == 'GET':
        return render(request, template_url, {'group': group})

    form = (GroupForm if group_id else NewGroupForm)(request.POST)

    if form.is_valid():
        try:
            groups_ctl.save_wiki_group(request.user,
                                       form.cleaned_data,
                                       group_id)
        except ObjectDoesNotExist:
            raise Http404
        except PermissionDenied:
            return HttpResponseForbidden()

        return HttpResponseRedirect(reverse('groups:list'))

    return render(request, template_url, {'form': form, 'group': group})


@require_http_methods(['POST'])
def group_deactivate_view(request, group_id):
    group_id = int(group_id)
    try:
        groups_ctl.deactivate_group(request.user, group_id)
    except ObjectDoesNotExist:
        raise Http404
    except PermissionDenied:
        return HttpResponseForbidden()

    return HttpResponseRedirect(reverse('groups:list'))


@require_http_methods(['GET'])
def group_view(request, group_url):
    try:
        group = Group.objects.get(url=group_url, intranet_status=1)
    except Group.DoesNotExist:
        raise Http404

    return render(
        request,
        'groups/view_group.html',
        {
            'group': group,
            'can_edit_wiki_group': can_edit_group(group, request.user),
            'can_delete_wiki_group': can_delete_group(request.user),
            'upravlyator_host': settings.UPRAVLYATOR_HOST,
        },
    )
