from itertools import chain
from typing import Dict, List, Optional, Sequence, Set

from staff.lib import waffle
from django.db.models import Q
from django.db.models.query import ValuesQuerySet

from staff.lib.utils.qs_values import extract_related, localize

from staff.budget_position.models import BudgetPositionComment, BudgetPositionAssignmentStatus
from staff.departments.models import Vacancy
from staff.departments.tree_lib import AbstractEntityInfo

from staff.headcounts.models import CreditManagementApplicationRow


class BudgetPositionAssignmentEntityInfo(AbstractEntityInfo):
    value_stream_field_name = 'value_stream'

    def __init__(self, filter_context):
        super().__init__(filter_context)
        self._enable_maternity_leave = waffle.switch_is_active('enable_maternity_leave')

    def fill_counters(self, all_ids, departments):
        pass

    def _is_occupied(self, position):
        return position['status'] == BudgetPositionAssignmentStatus.OCCUPIED.value

    def _is_offer(self, position):
        return position['status'] == BudgetPositionAssignmentStatus.OFFER.value

    def _is_vacancy_open(self, position):
        return position['status'] == BudgetPositionAssignmentStatus.VACANCY_OPEN.value

    def _is_vacancy_plan(self, position):
        return position['status'] == BudgetPositionAssignmentStatus.VACANCY_PLAN.value

    def _fill_user(self, position):
        position['current_person'] = self.user_data(position, 'person')
        position['previous_person'] = self.user_data(position, 'previous_assignment__person')

    def _extract_vacancies(self, positions_codes: Set[int]) -> Dict[int, Dict]:
        vacancies_qs = (
            Vacancy.objects
            .filter(headcount_position_code__in=positions_codes, is_active=True)
            .values()
        )
        return {vacancy['headcount_position_code']: vacancy for vacancy in vacancies_qs}

    def _extract_credit_applications(self, positions_codes: Set[int]) -> Dict[int, Dict]:
        applications_qs = (
            CreditManagementApplicationRow.objects
            .filter(
                Q(credit_budget_position__code__in=positions_codes)
                | Q(repayment_budget_position__code__in=positions_codes),
                application__is_active=True,
            )
            .values(
                'credit_budget_position__code',
                'repayment_budget_position__code',
                'application__author__login',
                'application__author__first_name',
                'application__author__last_name',
                'application__author__first_name_en',
                'application__author__last_name_en',
                'application__startrek_headcount_key',
            )
        )
        result = {}
        for application in applications_qs:
            application['author'] = self.user_data(application, 'application__author')
            result[application['credit_budget_position__code']] = application
            result[application['repayment_budget_position__code']] = application
        return result

    def fill_positions(self, positions: List[Dict]):
        pk_to_position = self._extract_positions_for_links(positions)

        all_considered_positions = positions + list(pk_to_position.values())
        positions_codes = {position['budget_position__code'] for position in all_considered_positions}
        positions_to_vacancies_map = self._extract_vacancies(positions_codes)
        positions_to_applications_map = self._extract_credit_applications(positions_codes)
        positions_comments: Optional[Dict[int, str]] = None

        if self.filter_context.observer_permissions:
            positions_comments = dict(
                BudgetPositionComment.objects
                .filter(person=self.filter_context.observer_permissions.person())
                .filter(budget_position__code__in=positions_codes)
                .values_list('budget_position__code', 'comment')
            )

        for position in all_considered_positions:
            position['headcount'] = position.pop('budget_position__headcount')

            self._fill_user(position)
            self._fill_vacancy(position, positions_to_vacancies_map)
            self._fill_credit_application(position, positions_to_applications_map)
            self._fill_category(position)
            self._fill_maternity_leave(position)
            self._fill_department_and_value_stream(position)
            self._fill_geography(position)
            self._fill_reward_category(position)
            self._fill_user_comments(position, positions_comments)
            position['is_crossing'] = position['next_assignment__id'] is not None
            position['code'] = position.pop('budget_position__code')
            position['category_is_new'] = position['creates_new_position']

        for position in positions:
            self._fill_common_positions(position, pk_to_position)
            self._fill_links(position)

    def fill_info(self, all_ids, departments, info_map):
        positions = list(chain.from_iterable(info_map.values()))
        self.fill_positions(positions)

        for dep in departments:
            dep['is_deleted'] = not dep['intranet_status']
            info = dep.setdefault('info', {})

            if info_map is not None and dep['id'] in info_map:
                info['positions'] = sorted(
                    info_map[dep['id']],
                    key=lambda k: (k['category'] == 'NEW', k['status'])
                )

        return departments

    def fill_dep_attrs(self, all_ids: Sequence, departments: Sequence) -> Sequence:
        return departments

    @staticmethod
    def user_data(data, user_prefix):
        user_data = extract_related(data, user_prefix)
        assert 'login' in user_data

        if not user_data['login']:
            return None

        return {
            'login': user_data['login'],
            'name': '{first_name} {last_name}'.format(**localize(user_data)),
        }

    def filter_result(self, departments):
        return departments

    def filter(self, all_ids, departments):
        return departments

    def _fill_category(self, position: Dict):
        category_text = 'NEW' if position['creates_new_position'] else 'REPLACEMENT'
        position['category'] = category_text

    def _status(self, position):
        return position['status']

    def _fill_department_and_value_stream(self, position):
        position['department'] = localize(extract_related(position, 'department__'))
        position['value_stream'] = localize(extract_related(position, 'value_stream__'))

    def _fill_geography(self, position) -> None:
        position['geography'] = localize(extract_related(position, 'geography__'))

    def _fill_reward_category(self, position) -> None:
        position['reward_category'] = position.pop('reward__category')

    def _fill_user_comments(self, position, positions_comments: Optional[Dict[int, str]]) -> None:
        if positions_comments is not None:
            position['comment'] = positions_comments.get(position['budget_position__code'])

    def _fill_vacancy(self, position, positions_to_vacancies_map: Dict[int, Dict]) -> None:
        vacancy: Optional[Dict] = positions_to_vacancies_map.get(position['budget_position__code'])

        if not vacancy:
            return

        keys = [
            'id',
            'name',
            'ticket',
            'preprofile_id',
            'candidate_first_name',
            'candidate_last_name',
            'candidate_login',
            'candidate_id',
        ]

        position['vacancy'] = {}
        for key in keys:
            position['vacancy'][key] = vacancy[key]

    def _fill_credit_application(self, position, positions_to_applications_map: Dict[int, Dict]) -> None:
        application: Optional[Dict] = positions_to_applications_map.get(position['budget_position__code'])

        position['is_disabled'] = application is not None

        if not application:
            return

        position['credit_application'] = {
            'author': application['author'],
            'ticket': application['application__startrek_headcount_key'],
        }

    def _fill_maternity_leave(self, position) -> None:
        position['is_maternity_leave_available'] = (
            self._enable_maternity_leave
            and self._has_special_role()
            and position['status'] == BudgetPositionAssignmentStatus.OCCUPIED.value
            and position['current_person'] is not None
        )

    def _has_special_role(self):
        observer_permissions = self.filter_context.observer_permissions
        if not observer_permissions:
            return False
        else:
            return observer_permissions.has_special_role()

    def _fill_links(self, position):
        position['link_new'] = self._is_vacancy_plan(position)
        position['link_replacement'] = self._is_occupied(position) and position['next_position'] is None
        position['link_changehc'] = (
            (self._is_vacancy_plan(position) or self._is_vacancy_open(position))
            and position['headcount'] > 0
        )

    def order_entities_by_fields(self) -> List[str]:
        return ['creates_new_position', 'status', 'budget_position__code']

    def _extract_positions_for_links(self, positions: List) -> Dict[int, Dict]:
        filters = list(chain(
            ({'pk': p['previous_assignment_id']} for p in positions if p['previous_assignment_id']),
            ({'pk': p['next_assignment__id']} for p in positions if p['next_assignment__id']),
        ))

        positions = self.filter_context.positions_for_links(filters)
        return {position['id']: position for position in positions}

    def _fill_common_positions(self, position: Dict, pk_to_position: Dict[int, Dict]) -> None:
        position['prev_position'] = pk_to_position.get(position.pop('previous_assignment_id'))
        position['next_position'] = pk_to_position.get(position.pop('next_assignment__id'))

    def fill_list(self, entities):
        return entities

    def departments_query(self) -> ValuesQuerySet:
        return self.filter_context.departments_qs().order_by('tree_id', 'lft')

    def entities_quantity_by_department_query(self, entity_dep_field_name):
        positions_quantity_qs = self.filter_context.positions_quantity_qs(entity_dep_field_name)
        field_name = entity_dep_field_name.replace('_id', '')
        positions_quantity_qs_ordered = positions_quantity_qs.order_by(f'{field_name}__tree_id', f'{field_name}__lft')
        return positions_quantity_qs_ordered

    def short_entities_query(self):
        return self.filter_context.positions_qs()

    def full_entities_query(self):
        return self.short_entities_query()

    def total_entities_count_query(self):
        return self.total_entities_aggregate_query(['budget_position__code'])

    def total_entities_aggregate_query(self, fields):
        return self.filter_context.positions_qs(fields)
