from operator import and_
from typing import List, Dict, Iterable, Optional

from django.conf import settings
from django.db.models import Count
from django.db.models.query import Q, QuerySet, ValuesQuerySet, ValuesListQuerySet

from staff.budget_position.models import BudgetPositionAssignment, BudgetPositionAssignmentStatus
from staff.departments.models import Department, Vacancy
from staff.headcounts.forms import QUANTITY_FILTER
from staff.headcounts.permissions import Permissions
from staff.lib.models.mptt import get_heirarchy_filter_query
from staff.lib.models.query import build_search_query


class BudgetPositionAssignmentFilterContext:
    budget_position_assignment_fields = (
        'id',
        'budget_position__code',
        'name',

        'department_id',
        'department__id',
        'department__url',
        'department__name',
        'department__name_en',

        'value_stream_id',
        'value_stream__id',
        'value_stream__url',
        'value_stream__name',
        'value_stream__name_en',

        'geography_id',
        'geography__id',
        'geography__url',
        'geography__name',
        'geography__name_en',

        'budget_position__headcount',
        'creates_new_position',

        'person__login',
        'person__first_name',
        'person__last_name',
        'person__first_name_en',
        'person__last_name_en',

        'status',
        'main_assignment',
        'replacement_type',

        'previous_assignment__person__login',
        'previous_assignment__person__first_name',
        'previous_assignment__person__last_name',
        'previous_assignment__person__first_name_en',
        'previous_assignment__person__last_name_en',

        'previous_assignment_id',
        'next_assignment__id',
        'reward_id',
        'reward__category',
        'change_reason',
    )

    dep_fields = (
        'id',
        'parent_id',
        'url',
        'tree_id',
        'lft',
        'rght',
        'name',
        'name_en',
        'level',
        'description',
        'description_en',
        'wiki_page',
        'maillists',
        'clubs',
        'intranet_status',
        'instance_class',
    )

    filter_id = None

    def __init__(self, observer_permissions: Permissions = None, exclude_reserve=True, **kwargs):
        self.observer_permissions = observer_permissions

        self.filter_q = Q()

        if kwargs.get('position_statuses'):
            self.filter_q &= Q(status__in=kwargs['position_statuses'])

        if kwargs.get('category_is_new') is not None:
            self.filter_q &= Q(creates_new_position=kwargs['category_is_new'])

        if kwargs.get('is_crossing') is not None:
            self.filter_q &= Q(next_assignment__isnull=not kwargs['is_crossing'])

        if kwargs.get('main_assignment') is not None:
            self.filter_q &= Q(main_assignment=kwargs['main_assignment'])

        if kwargs.get('replacement_type'):
            self.filter_q &= Q(replacement_type__in=kwargs['replacement_type'])

        if kwargs.get('quantity'):
            quantity = kwargs['quantity']
            if quantity == QUANTITY_FILTER.ZERO:
                self.filter_q &= Q(budget_position__headcount=0)
            elif quantity == QUANTITY_FILTER.GREATER_THAN_ZERO:
                self.filter_q &= Q(budget_position__headcount__gt=0)
            elif quantity == QUANTITY_FILTER.LESS_THAN_ZERO:
                self.filter_q &= Q(budget_position__headcount__lt=0)

        if kwargs.get('current_person'):
            self.filter_q &= Q(person=kwargs['current_person'])

        if kwargs.get('replaced_person'):
            self.filter_q &= Q(previous_assignment__person=kwargs['replaced_person'])

        if kwargs.get('code') is not None:
            self.filter_q &= Q(budget_position__code=kwargs['code'])

        if kwargs.get('codes'):
            self.filter_q &= Q(budget_position__code__in=kwargs['codes'])

        if kwargs.get('department_url'):
            self.filter_q &= Q(department__url=kwargs['department_url'])

        if kwargs.get('department'):
            self.filter_q &= get_heirarchy_filter_query(
                mptt_objects=[kwargs['department']],
                by_children=True,
                filter_prefix='department__',
                include_self=True,
            )

        if kwargs.get('value_stream'):
            self.filter_q &= get_heirarchy_filter_query(
                mptt_objects=[kwargs['value_stream']],
                by_children=True,
                filter_prefix='value_stream__',
                include_self=True,
            )

        if kwargs.get('geography'):
            self.filter_q &= get_heirarchy_filter_query(
                mptt_objects=[kwargs['geography']],
                by_children=True,
                filter_prefix='geography__',
                include_self=True,
            )

        if kwargs.get('search_text'):
            text = kwargs['search_text']
            self.filter_q &= self._get_qs_for_searching_by_all_details(text)

        if kwargs.get('category'):
            self.filter_q &= Q(reward__category__in=kwargs['category'])

        if exclude_reserve:
            self.filter_q &= ~(
                Q(status=BudgetPositionAssignmentStatus.RESERVE.value) & Q(budget_position__headcount__gte=0)
            )

    def _get_qs_for_searching_by_all_details(self, text: str) -> Q:
        occupied_status = Q(status=BudgetPositionAssignmentStatus.OCCUPIED.value)
        offer_status = Q(status=BudgetPositionAssignmentStatus.OFFER.value)
        vacancy_status = Q(status=BudgetPositionAssignmentStatus.VACANCY_OPEN.value)

        # TODO: Remove
        free_status = Q(
            status__in=(
                BudgetPositionAssignmentStatus.VACANCY_PLAN.value,
                BudgetPositionAssignmentStatus.MATERNITY.value,
            ),
        )

        person_fields = [
            'person__login',
            'person__first_name',
            'person__last_name',
            'person__first_name_en',
            'person__last_name_en',
        ]

        person_query = build_search_query(text, person_fields, and_)
        vacancies_qs = self._get_base_vacancies_qs()

        offer_fields = ['candidate_first_name', 'candidate_last_name']
        offer_query = vacancies_qs.filter(build_search_query(text, offer_fields, and_))

        vacancy_query = vacancies_qs.filter(name__icontains=text)
        ticket_query = vacancies_qs.filter(self._ticket_q(text))

        return (
            person_query & occupied_status
            | (
                Q(name__icontains=text)
                & (free_status | (offer_status | vacancy_status) & ~Q(budget_position__code__in=vacancies_qs))
            )
            | Q(budget_position__code__in=offer_query) & offer_status
            | Q(budget_position__code__in=vacancy_query) & vacancy_status
            | Q(budget_position__code__in=ticket_query) & (offer_status | vacancy_status)
        )

    @staticmethod
    def _ticket_q(search_text: str) -> Q:
        ticket_query = Q()

        for part in search_text.split():
            ticket_query |= Q(ticket__icontains=part)

        return ticket_query

    @staticmethod
    def _get_base_vacancies_qs() -> ValuesListQuerySet:
        vacancies_qs = (
            Vacancy.objects
            .filter(is_active=True, headcount_position_code__isnull=False)
            .values_list('headcount_position_code')
        )
        return vacancies_qs

    def departments_qs(self, fields: Optional[Iterable[str]] = None) -> ValuesQuerySet:
        if fields is None:
            fields = self.dep_fields

        return (
            Department.all_types
            .annotate(Count('positions'))
            .filter(Q(intranet_status=1) | Q(positions__count__gt=0))
            .exclude(id=settings.INTRANET_DISMISSED_STAFF_DEPARTMENT_ID)
            .values(*fields)
        )

    def positions_objects_qs(self) -> QuerySet:
        qs = (
            self.observer_permissions.available_budget_position_assignments()
            if self.observer_permissions
            else BudgetPositionAssignment.objects.active()
        )
        return qs.filter(self.filter_q)

    def positions_qs(self, fields: Iterable[str] = None) -> ValuesQuerySet:
        fields = fields or self.budget_position_assignment_fields
        return self.positions_objects_qs().values(*fields)

    def positions_quantity_qs(self, entity_dep_field_name) -> ValuesQuerySet:
        return self.positions_qs().values(entity_dep_field_name).annotate(qty=Count('budget_position__code'))

    def positions_for_links(self, or_filters: List[Dict]) -> List[Dict]:
        if not or_filters:
            return []

        filter_qs = Q(id=None)
        for filter_ in or_filters:
            filter_qs |= Q(**filter_)

        assert self.observer_permissions
        by_department_filter_q = Q(
            department_id__in=self.observer_permissions.filter_by_observer(Department.objects.all())
        )

        by_value_stream_filter_q = Q(
            value_stream_id__in=self.observer_permissions.filter_by_observer(Department.valuestreams.all())
        )

        qs = (
            BudgetPositionAssignment.objects
            .active()
            .filter(filter_qs)
            .filter(by_department_filter_q | by_value_stream_filter_q)
            .exclude(status=BudgetPositionAssignmentStatus.RESERVE.value)
            .values(*self.budget_position_assignment_fields)
        )

        return list(qs)
