from typing import List, Optional

from django.conf import settings
from django.template import loader


from staff.departments.models import Department
from staff.lib.startrek.issues import create_issue

from staff.headcounts.headcounts_credit_management import use_cases


class Startrek(use_cases.Startrek):
    def _ancestors_chain(self, dep_url: Optional[str]) -> str:
        if not dep_url:
            return ''

        return ' / '.join(
            ancestor.name
            for ancestor in Department.objects.get(url=dep_url).get_ancestors(include_self=True)
        )

    def _vacancy_url(self, id):
        return f'((https://{settings.FEMIDA_HOST}/vacancies/{id}/ VAC {id}))' if id else None

    def _transaction_url(self, id: int):
        return f'((https://{settings.STAFF_HOST}/admin/budget_position/workflow/?credit_management_id__exact={id}))'

    def _description(self, credit_repayment: use_cases.CreditRepayment):
        context = {
            'comment': credit_repayment.comment,
            'transaction_url': self._transaction_url(credit_repayment.id),
            'application': [
                {
                    'credit_bp_code': row.credit_budget_position.code,
                    'credit_bp_department': self._ancestors_chain(row.credit_budget_position.department_url),
                    'credit_bp_vs': row.credit_budget_position.valuestream or '',
                    'credit_bp_geo': row.credit_budget_position.geography or None,
                    'repayment_bp_code': row.repayment_budget_position.code,
                    'repayment_bp_department': self._ancestors_chain(row.repayment_budget_position.department_url),
                    'repayment_bp_vs': row.repayment_budget_position.valuestream or '',
                    'repayment_bp_geo': row.repayment_budget_position.geography or None,
                    'vacancy_url': self._vacancy_url(row.repayment_budget_position.vacancy_id),
                }
                for row in credit_repayment.rows
            ]
        }

        return loader.render_to_string(template_name='startrek/credit_management.html', context=context)

    def _collect_followers(self, rows: List[use_cases.CreditRepaymentRow]) -> List[str]:
        return []

    def create_ticket(self, credit_repayment: use_cases.CreditRepayment) -> str:
        followers = self._collect_followers(credit_repayment.rows)
        description = self._description(credit_repayment)

        issue = create_issue(
            queue=settings.HEADCOUNT_QUEUE_ID,
            createdBy=credit_repayment.author_login,
            summary='Погашение кредитной позиции',
            description=description,
            followers=followers,
            unique=f'credit_repayment_{credit_repayment.id}_{settings.STAFF_HOST}',
        )

        return issue.key
