from django.conf import settings

from staff.budget_position.models import BudgetPositionAssignmentStatus, ReplacementType
from staff.lib import xlsx

from staff.headcounts.views.utils import find


class FlexiblePositionsSheetPresenter(xlsx.SheetPresenter):
    heading = (
        'Подразделение',
        'Иерархия подразделения',
        'Основной продукт\n (Value Stream)',
        'Иерархия основного продукта\n (Value Stream)',
        'География',
        'Категория',
        'Статус позиции',
        'Детали позиции',
        'ФИО',
        'Фемида',
        'Кому замена',
        'ФИО',
        'Численность',
        'Тип пересечения',
        'Есть замена',
        'ФИО',
        'ХК еще занят',
        'ФИО',
        'Номер БП',
        'Подразделение удалено',
        'ID подразделения',
    )

    _gray_row_format = {
        'font_color': '#D3D3D3',
    }

    _heading_format = {
        'align': 'center',
        'valign': 'vcenter',
        'text_wrap': True,
        'bold': True,
        'bg_color': '#FFCC00',
    }

    _without_outline_options = {'level': 0}
    _within_outline_options = {'level': 1, 'hidden': True}
    _collapsed_options = {'level': 0, 'collapsed': True}

    def columns(self):
        return [
            xlsx.Column(width=25, options=self._without_outline_options),
            xlsx.Column(width=12, options=self._without_outline_options),
            xlsx.Column(width=25, options=self._without_outline_options),
            xlsx.Column(width=25, options=self._without_outline_options),
            xlsx.Column(width=19, options=self._without_outline_options),
            xlsx.Column(width=13, options=self._without_outline_options),
            xlsx.Column(width=19, options=self._without_outline_options),
            xlsx.Column(width=16, options=self._without_outline_options),
            xlsx.Column(width=16, options=self._without_outline_options),
            xlsx.Column(width=13, options=self._without_outline_options),
            xlsx.Column(width=13, options=self._collapsed_options),
            xlsx.Column(width=13, options=self._within_outline_options),
            xlsx.Column(width=13, options=self._without_outline_options),
            xlsx.Column(width=22, options=self._without_outline_options),
            xlsx.Column(width=22, options=self._collapsed_options),
            xlsx.Column(width=22, options=self._within_outline_options),
            xlsx.Column(width=16, options=self._collapsed_options),
            xlsx.Column(width=16, options=self._within_outline_options),
            xlsx.Column(width=17, options=self._without_outline_options),
            xlsx.Column(width=15, options=self._without_outline_options),
            xlsx.Column(width=8.43, options=self._without_outline_options),
        ]

    def column_caption_row(self):
        return xlsx.ViewRow(
            xlsx.ViewCell(caption, self._heading_format)
            for caption in self.heading
        )

    def rows(self):
        yield self.column_caption_row()

        for row in self._model:
            yield xlsx.ViewRow(
                cells=self.default_view_cells_from_row_data(self.data_row_to_view_item(row)),
                row_format=self._row_format(row)
            )

    def _row_format(self, data_row):
        return None if data_row['department__intranet_status'] else self._gray_row_format

    @staticmethod
    def _display_chain(chain):
        return ' → '.join(dep['name'] for dep in chain)

    @staticmethod
    def data_row_to_view_item(data_row):
        value_stream_name = find('value_stream.name', data_row) or ''
        value_stream_chain = find('value_stream.chain', data_row)
        if value_stream_chain:
            value_stream_chain = FlexiblePositionsSheetPresenter._display_chain(value_stream_chain)
        else:
            value_stream_chain = ''
        geography = find('geography.name', data_row) or ''
        return (
            data_row['department']['name'],
            FlexiblePositionsSheetPresenter._display_chain(data_row['department']['chain']),
            value_stream_name,
            value_stream_chain,
            geography,
            data_row['reward_category'] or '',
            FlexiblePositionsSheetPresenter._get_text_for_status_column(data_row['status']),
            FlexiblePositionsSheetPresenter._get_position_name(data_row),
            FlexiblePositionsSheetPresenter._get_position_full_name(data_row),
            FlexiblePositionsSheetPresenter._get_vacancy_cell(data_row)
            or FlexiblePositionsSheetPresenter._get_offer_cell(data_row),
            find('previous_person.login', data_row) or '',
            FlexiblePositionsSheetPresenter._get_prev_user_fullname(data_row),
            data_row['headcount'],
            FlexiblePositionsSheetPresenter._get_replacement_type(data_row['replacement_type']),
            FlexiblePositionsSheetPresenter._get_next_position_name(data_row),
            FlexiblePositionsSheetPresenter._get_next_position_full_name(data_row),
            FlexiblePositionsSheetPresenter._get_prev_position_name(data_row),
            FlexiblePositionsSheetPresenter._get_prev_position_full_name(data_row),
            data_row['code'],
            'Нет' if data_row['department__intranet_status'] else 'Да',
            data_row['department']['id'],
        )

    @staticmethod
    def _get_prev_user_fullname(position):
        user = position.get('previous_person')
        if user:
            return user.get('name')
        return ''

    @staticmethod
    def _get_position_full_name(position):
        if not position:
            return 'Нет доступа'
        current_person = position.get('current_person')
        if current_person:
            full_name = current_person['name']
            if full_name != 'None':
                return full_name
        return ''

    @staticmethod
    def _get_position_name(position):
        if not position:
            return 'Нет доступа'
        return (
            find('current_person.login', position)
            or FlexiblePositionsSheetPresenter._get_ticket_cell(position)
            or position['name']
        )

    @staticmethod
    def _get_vacancy_cell(position):
        if position['status'] == BudgetPositionAssignmentStatus.VACANCY_OPEN.value and position.get('vacancy'):
            return xlsx.ViewCell(
                text=position['vacancy']['name'],
                url='https://{}/vacancies/{}/'.format(settings.FEMIDA_HOST, position['vacancy']['id']),
            )

        return ''

    @staticmethod
    def _get_offer_cell(position):
        if position['status'] == BudgetPositionAssignmentStatus.OFFER.value and position.get('vacancy'):
            return xlsx.ViewCell(
                text='{} {}'.format(
                    position['vacancy']['candidate_first_name'],
                    position['vacancy']['candidate_last_name'],
                ),
                url='https://{}/candidates/{}/'.format(settings.FEMIDA_HOST, position['vacancy']['candidate_id']),
            )
        return ''

    @staticmethod
    def _get_ticket_cell(position):
        available_statuses = [
            BudgetPositionAssignmentStatus.VACANCY_OPEN.value,
            BudgetPositionAssignmentStatus.OFFER.value,
        ]
        if position['status'] in available_statuses and position.get('vacancy', {}).get('ticket'):
            return xlsx.ViewCell(
                text=position['vacancy']['ticket'],
                url='{}/{}'.format(settings.STARTREK_URL, position['vacancy']['ticket']),
            )
        return ''

    @staticmethod
    def _get_next_position_name(position):
        available_types = [ReplacementType.HAS_REPLACEMENT.value, ReplacementType.HAS_REPLACEMENT_AND_BUSY.value]
        if position['replacement_type'] in available_types:
            return FlexiblePositionsSheetPresenter._get_position_name(position['next_position'])
        return ''

    @staticmethod
    def _get_prev_position_name(position):
        available_types = [ReplacementType.BUSY.value, ReplacementType.HAS_REPLACEMENT_AND_BUSY.value]
        if position['replacement_type'] in available_types:
            return FlexiblePositionsSheetPresenter._get_position_name(position['prev_position'])
        return ''

    @staticmethod
    def _get_next_position_full_name(position):
        available_types = [ReplacementType.HAS_REPLACEMENT.value, ReplacementType.HAS_REPLACEMENT_AND_BUSY.value]
        if position['replacement_type'] in available_types:
            return FlexiblePositionsSheetPresenter._get_position_full_name(position['next_position'])
        return ''

    @staticmethod
    def _get_prev_position_full_name(position):
        available_types = [ReplacementType.BUSY.value, ReplacementType.HAS_REPLACEMENT_AND_BUSY.value]
        if position['replacement_type'] in available_types:
            return FlexiblePositionsSheetPresenter._get_position_full_name(position['prev_position'])
        return ''

    @staticmethod
    def _get_text_for_status_column(value):
        statuses_map = {
            BudgetPositionAssignmentStatus.OCCUPIED.value: 'Работает',
            BudgetPositionAssignmentStatus.OFFER.value: 'Оффер',
            BudgetPositionAssignmentStatus.VACANCY_OPEN.value: 'Открытая вакансия',
            BudgetPositionAssignmentStatus.VACANCY_PLAN.value: 'Незанятый хедкаунт',
            BudgetPositionAssignmentStatus.RESERVE.value: 'Кредит',
            BudgetPositionAssignmentStatus.MATERNITY.value: 'Декрет',
        }

        return statuses_map.get(value, value)

    @staticmethod
    def _get_replacement_type(value):
        types = {
            ReplacementType.HAS_REPLACEMENT.value: 'Есть замена',
            ReplacementType.BUSY.value: 'ХК еще занят',
            ReplacementType.HAS_REPLACEMENT_AND_BUSY.value: 'Есть замена и ХК еще занят',
            ReplacementType.WITHOUT_REPLACEMENT.value: 'Нет пересечений',
        }

        return types.get(value, value)
