from itertools import chain
from typing import Any, Dict, List, Optional

import attr

from django.db.models import Q, QuerySet

from staff.budget_position.models import BudgetPositionAssignment, BudgetPositionAssignmentStatus, ReplacementType
from staff.departments.models import Department as DepartmentModel

from staff.headcounts.headcounts_summary.query_builder import QueryParams, RelatedEntity


@attr.s(auto_attribs=True)
class Name:
    name_en: str
    name_ru: str


@attr.s(auto_attribs=True)
class Vacancy:
    id: int
    name: str
    candidate_id: Optional[int]
    candidate_first_name: Optional[str]
    candidate_last_name: Optional[str]
    ticket: str


@attr.s(auto_attribs=True)
class Department:
    id: int
    intranet_status: int
    name: Name


@attr.s(auto_attribs=True)
class RowModel:
    budget_position_assignment_id: int
    headcount: int
    code: int
    department: Department
    department_chain: List[Department]
    value_stream: Department
    value_stream_chain: List[Department]
    geography: Department
    category: Optional[str]
    person_first_name: Optional[Name]
    person_last_name: Optional[Name]
    person_login: Optional[str]
    vacancy: Optional[Vacancy]
    status: BudgetPositionAssignmentStatus
    name: Optional[str]
    replacement_type: ReplacementType
    next_budget_position_assignment_id: Optional[int]
    next_budget_position_assignment: Optional['RowModel']
    prev_budget_position_assignment_id: Optional[int]
    prev_budget_position_assignment: Optional['RowModel']


class AssignmentsPageModel:
    def __init__(self, query_params: QueryParams) -> None:
        self._query_params = query_params

    def _field_prefix_for_filter(self, related_entity: RelatedEntity) -> str:
        return related_entity.value

    def _make_query_for_related_entity(self, related_entity: RelatedEntity) -> Q:
        perm_filter, user_filter = self._query_params.filters_pair_for_grouping(related_entity)
        result = Q()
        for filter in chain(perm_filter, user_filter):
            result &= Q(**{
                f'{self._field_prefix_for_filter(related_entity)}__lft__gte': filter.lft,
                f'{self._field_prefix_for_filter(related_entity)}__rght__lte': filter.rght,
                f'{self._field_prefix_for_filter(related_entity)}__tree_id': filter.tree_id,
            })

        return result

    def _make_filter_query(self) -> Q:
        query_filter = Q()
        for related_entity in (RelatedEntity.department, RelatedEntity.value_stream, RelatedEntity.geography):
            query_filter |= self._make_query_for_related_entity(related_entity)

        return query_filter

    def _make_order_by(self) -> List[str]:
        result = []
        for related_entity in self._query_params.groupings:
            result.append(f'{self._field_prefix_for_filter(related_entity)}__tree_id')
            result.append(f'{self._field_prefix_for_filter(related_entity)}__lft')

        return result

    def _budget_position_assignments_qs(self) -> QuerySet:
        fields = (
            'id',
            'department__id',
            'department__name',
            'department__name_en',
            'department__intranet_status',
            'value_stream__id',
            'value_stream__name',
            'value_stream__name_en',
            'value_stream__intranet_status',
            'geography__id',
            'geography__name',
            'geography__name_en',
            'geography__intranet_status',
            'reward__category',
            'previous_assignment_id',
            'next_assignment__id',
            'replacement_type',
            'person__login',
            'person__first_name',
            'person__first_name_en',
            'person__last_name',
            'person__last_name_en',
            'status',
            'name',
            'budget_position__headcount',
            'budget_position__code',
            'budget_position__vacancy__id',
            'budget_position__vacancy__ticket',
            'budget_position__vacancy__name',
            'budget_position__vacancy__candidate_id',
            'budget_position__vacancy__candidate_first_name',
            'budget_position__vacancy__candidate_last_name',
        )
        return (
            BudgetPositionAssignment.objects
            .filter(self._make_filter_query())
            .order_by(*self._make_order_by())
            .values(*fields)
        )

    def _as_department(self, row: Dict[str, Any], prefix: str) -> Department:
        return Department(
            id=row[f'{prefix}id'],
            name=Name(name_en=row[f'{prefix}name_en'], name_ru=row[f'{prefix}name']),
            intranet_status=row[f'{prefix}intranet_status'],
        )

    def _as_row_model(self, row: Dict[str, Any]) -> RowModel:
        department = self._as_department(row, 'department__')
        value_stream = self._as_department(row, 'value_stream__')
        geography = self._as_department(row, 'geography__')
        vacancy = (
            Vacancy(
                id=row['budget_position__vacancy__id'],
                name=row['budget_position__vacancy__name'],
                candidate_first_name=row['budget_position__vacancy__candidate_first_name'],
                candidate_last_name=row['budget_position__vacancy__candidate_last_name'],
                candidate_id=row['budget_position__vacancy__candidate_id'],
                ticket=row['budget_position__vacancy__ticket'],
            )
            if row['budget_position__vacancy__id']
            else None
        )
        person_first_name: Optional[Name] = None
        person_last_name: Optional[Name] = None
        login = row['person__login']

        if login:
            person_first_name = Name(row['person__first_name_en'], row['person__first_name'])
            person_last_name = Name(row['person__last_name_en'], row['person__last_name'])

        return RowModel(
            budget_position_assignment_id=row['id'],
            department=department,
            department_chain=[],
            value_stream=value_stream,
            value_stream_chain=[],
            geography=geography,
            category=row['reward__category'],
            person_first_name=person_first_name,
            person_last_name=person_last_name,
            person_login=login,
            status=BudgetPositionAssignmentStatus(row['status']),
            name=row['name'],
            vacancy=vacancy,
            replacement_type=ReplacementType(row['replacement_type']),
            next_budget_position_assignment_id=row['next_assignment__id'],
            next_budget_position_assignment=None,
            prev_budget_position_assignment_id=row['previous_assignment_id'],
            prev_budget_position_assignment=None,
            headcount=row['budget_position__headcount'],
            code=row['budget_position__code'],
        )

    def _department_chains(self) -> Dict[int, List[Department]]:
        qs = (
            DepartmentModel.all_types
            .values('id', 'parent_id', 'name', 'name_en', 'intranet_status')
            .order_by('tree_id', 'lft')
        )

        result = {}

        for dep_row in qs:
            current_department = Department(
                id=dep_row['id'],
                intranet_status=dep_row['intranet_status'],
                name=Name(dep_row['name_en'], dep_row['name']),
            )
            if dep_row['parent_id'] is None:
                result[dep_row['id']] = [current_department]
            else:
                result[dep_row['id']] = result[dep_row['parent_id']] + [current_department]

        return result

    def get_result(self) -> List[RowModel]:
        result = list(self._as_row_model(assignment) for assignment in self._budget_position_assignments_qs())
        by_id = {row.budget_position_assignment_id: row for row in result}
        chains = self._department_chains()
        for row in result:
            row.department_chain = chains[row.department.id]
            row.value_stream_chain = chains[row.value_stream.id]
            if row.prev_budget_position_assignment_id is not None:
                row.prev_budget_position_assignment = by_id.get(row.prev_budget_position_assignment_id)
            if row.next_budget_position_assignment_id is not None:
                row.next_budget_position_assignment = by_id.get(row.next_budget_position_assignment_id)

        return result
