from typing import Dict, Set

from staff.departments.models import DepartmentStaff, DepartmentRoles, ValuestreamRoles
from staff.lib.utils.qs_values import extract_related

from staff.headcounts.headcounts_summary.query_builder import query_results


class ChiefsInfoUpdater:
    _fields = (
        'department_id',
        'staff__first_name',
        'staff__first_name_en',
        'staff__last_name',
        'staff__last_name_en',
        'staff__login',
    )

    def _fetch_chiefs(self, ids: Set[int]) -> Dict[int, Dict[str, str]]:
        result = {}
        qs = (
            DepartmentStaff.objects
            .filter(department_id__in=ids, role_id__in=[DepartmentRoles.CHIEF.value, ValuestreamRoles.HEAD.value])
            .values(*self._fields)
        )

        for row in qs:
            result[row['department_id']] = extract_related(row, 'staff')

        return result

    def _gather_department_instances_id(self, results: query_results.Result) -> Set[int]:
        result = set()
        if results.grouping_instance_info:
            result.add(results.grouping_instance_info.id)
        for nested_result in results.next_level_grouping.values():
            result |= self._gather_department_instances_id(nested_result)
        for nested_result in results.children.values():
            result |= self._gather_department_instances_id(nested_result)
        return result

    def _set_chiefs_in_result(self, chiefs: Dict[int, Dict[str, str]], results: query_results.Result) -> None:
        if results.grouping_instance_info:
            if results.grouping_instance_info.id in chiefs:
                chief_info = chiefs[results.grouping_instance_info.id]
                results.grouping_instance_info.chief = query_results.ChiefInfo(
                    first_name=chief_info['first_name'],
                    last_name=chief_info['last_name'],
                    first_name_en=chief_info['first_name_en'],
                    last_name_en=chief_info['last_name_en'],
                    login=chief_info['login'],
                )

        for nested_result in results.next_level_grouping.values():
            self._set_chiefs_in_result(chiefs, nested_result)

        for nested_result in results.children.values():
            self._set_chiefs_in_result(chiefs, nested_result)

    def set_chiefs_into_result(self, results: query_results.Result) -> None:
        ids = self._gather_department_instances_id(results)
        chiefs = self._fetch_chiefs(ids)
        self._set_chiefs_in_result(chiefs, results)
