from copy import deepcopy
from typing import Iterable, List, Optional

import attr

from staff.headcounts.headcounts_summary.query_builder import Result, RelatedEntity


@attr.s(auto_attribs=True)
class Name:
    name: str
    name_en: str


@attr.s(auto_attribs=True)
class Grouping:
    related_entity: RelatedEntity
    hierarchy: List[Name]


@attr.s(auto_attribs=True)
class SummaryModelRow:
    hierarchy_by_groupings: List[Grouping]
    level: int
    name: Optional[Name]
    headcount: int
    working: int
    offers: int
    balance: int
    vacancy: int
    vacancy_plan: int
    vacancy_plan_new: int
    vacancy_plan_replacement: int
    credit: int
    crossing: int
    working_crossing: int
    offer_crossing: int
    vacancy_crossing: int
    vacancy_plan_crossing: int
    intranet_status: int

    @staticmethod
    def create_empty() -> 'SummaryModelRow':
        return SummaryModelRow(
            hierarchy_by_groupings=[],
            level=0,
            name=None,
            headcount=0,
            working=0,
            offers=0,
            balance=0,
            vacancy=0,
            vacancy_plan=0,
            vacancy_plan_new=0,
            vacancy_plan_replacement=0,
            credit=0,
            crossing=0,
            working_crossing=0,
            offer_crossing=0,
            vacancy_crossing=0,
            vacancy_plan_crossing=0,
            intranet_status=0,
        )


def export_summary_model(result: Optional[Result], user_groupings: List[RelatedEntity]) -> Iterable[SummaryModelRow]:
    if not result:
        return [SummaryModelRow.create_empty()]
    return _export_summary_model(result, 0, user_groupings, [])


def _export_summary_model(
    result: Result,
    level: int,
    user_groupings: List[RelatedEntity],
    hierarchy_by_groupings: List[Grouping],
) -> Iterable[SummaryModelRow]:
    current_name = (
        Name(name=result.grouping_instance_info.name, name_en=result.grouping_instance_info.name_en)
        if result.grouping_instance_info
        else None
    )
    yield SummaryModelRow(
        hierarchy_by_groupings=hierarchy_by_groupings,
        level=level,
        intranet_status=1,
        name=current_name,
        headcount=result.summary.headcount,
        working=result.summary.working,
        offers=result.summary.offer,
        balance=result.summary.balance,
        vacancy=result.summary.vacancies,
        vacancy_plan=result.summary.vacancies_plan_new + result.summary.vacancies_plan_replacement,
        vacancy_plan_new=result.summary.vacancies_plan_new,
        vacancy_plan_replacement=result.summary.vacancies_plan_replacement,
        credit=result.summary.credit,
        crossing=(
            result.summary.working_crossing
            + result.summary.vacancies_crossing
            + result.summary.offer_crossing
            + result.summary.vacancies_plan_crossing
        ),
        working_crossing=result.summary.working_crossing,
        offer_crossing=result.summary.offer_crossing,
        vacancy_crossing=result.summary.vacancies_crossing,
        vacancy_plan_crossing=result.summary.vacancies_plan_crossing,
    )

    for next_level_grouping_result in result.next_level_grouping.values():
        assert next_level_grouping_result.grouping_instance_info
        next_level_name = Name(
            name=next_level_grouping_result.grouping_instance_info.name,
            name_en=next_level_grouping_result.grouping_instance_info.name_en,
        )
        grouping_for_next_level = Grouping(related_entity=user_groupings[0], hierarchy=[next_level_name])
        yield from _export_summary_model(
            result=next_level_grouping_result,
            level=level + 1,
            user_groupings=user_groupings[1:],
            hierarchy_by_groupings=hierarchy_by_groupings + [grouping_for_next_level],
        )

    for child in result.children.values():
        hierarchy_for_next_children = deepcopy(hierarchy_by_groupings)
        assert child.grouping_instance_info
        child_name = Name(name=child.grouping_instance_info.name, name_en=child.grouping_instance_info.name_en)
        hierarchy_for_next_children[-1].hierarchy.append(child_name)
        yield from _export_summary_model(child, level + 1, user_groupings, hierarchy_for_next_children)
