from typing import Any, Dict, Tuple, List, Optional

import attr

from django.db import connections, router

from staff.departments.models import Department

from staff.headcounts.headcounts_summary.chiefs_info_updater import ChiefsInfoUpdater
from staff.headcounts.headcounts_summary.department_info_updater import DepartmentInfoUpdater
from staff.headcounts.headcounts_summary.empty_summary_filter import EmptySummaryFilter
from staff.headcounts.headcounts_summary.hierarchy_levels_filter import HierarchyLevelsFilter
from staff.headcounts.headcounts_summary.hierarchy_summary_counter import HierarchySummaryCounter
from staff.headcounts.headcounts_summary.query_builder import (
    Aliases,
    QueryBuilder,
    Result,
    ResultsMapper,
)
from staff.headcounts.headcounts_summary.related_entity_chains_builder import RelatedEntityChainsBuilder


class HeadcountsSummaryCalculator:
    static_fields = [
        'headcount',
        'working',
        'offer',
        'balance',
        'vacancies',
        'vacancies_plan_new',
        'vacancies_plan_replacement',
        'credit',
        'working_crossing',
        'offer_crossing',
        'vacancies_plan_crossing',
        'vacancies_crossing',
        'crossing',
    ]

    def __init__(self, query_builder: QueryBuilder):
        self._aliases = Aliases()
        self._query_builder = query_builder
        self._grouping_fields = [
            grouping.pk_field_name
            for grouping in query_builder.query_params.groupings
        ]
        self._query_fields = self._grouping_fields + self.static_fields
        self._chains_builder = RelatedEntityChainsBuilder(self._query_builder.query_params, self._aliases)
        self.query_params = query_builder.query_params

    def get_results(self) -> Optional[Result]:
        cursor = connections[router.db_for_read(Department)].cursor()
        query = self._query_builder.build()
        cursor.execute(query, tuple())
        values = self._fetch(cursor)
        if not values:
            return None

        chains = self._chains_builder.create_chains()
        mapper = ResultsMapper(self._query_builder.query_params.groupings, self._aliases, chains)
        mapper.map_to_result(values)
        results = mapper.root
        DepartmentInfoUpdater().set_departments_info_into_result(results)
        if self._query_builder.query_params.summary_with_child:
            HierarchySummaryCounter().summarize(results)
        if self._query_builder.query_params.enable_hierarchy_filter:
            HierarchyLevelsFilter(self._query_builder.query_params).filter(results)
        if self._query_builder.query_params.enable_empty_summary_filter:
            EmptySummaryFilter(self._query_builder.query_params).filter(results)
        ChiefsInfoUpdater().set_chiefs_into_result(results)
        return results

    def debug_summary(self) -> Dict[str, Any]:
        params = self._query_builder.query_params
        return {
            'query': self._query_builder.build(),
            'restrict_selection_to_departments': [attr.asdict(it) for it in params.department_permission_filters],
            'restrict_selection_to_value_streams': [attr.asdict(it) for it in params.value_stream_permission_filters],
            'restrict_selection_to_geographies': [attr.asdict(it) for it in params.geography_permission_filters],
        }

    def _fetch(self, cursor: Any) -> List[Dict[str, Any]]:
        rows = cursor.fetchall()
        # kludge to catch case when no rows for grouping operation, so grouping returns only one row with all Nulls
        if len(rows) == 1 and all(it is None for it in rows[0]):
            return []
        result = [self._map_row(row) for row in rows]
        return result

    def _map_row(self, row: Tuple[Any, ...]) -> Dict[str, Any]:
        return dict(zip(self._query_fields, row))
