from typing import List, Optional, Tuple

import attr

from staff.lib.utils.enum_choices import EnumChoices


class RelatedEntity(EnumChoices):
    department = 'department'
    value_stream = 'value_stream'
    geography = 'geography'

    @property
    def pk_field_name(self) -> str:
        return f'{self.value}_id'


@attr.s(auto_attribs=True)
class HierarchyValues:
    tree_id: int
    lft: int
    rght: int
    id: int

    @staticmethod
    def none() -> 'HierarchyValues':
        return HierarchyValues(tree_id=-1, lft=-1, rght=-1, id=-1)

    def is_child(self, other: 'HierarchyValues') -> bool:
        return other.tree_id == self.tree_id and other.lft < self.lft and other.rght > self.rght


@attr.s(auto_attribs=True)
class QueryParams:
    department_filters: List[HierarchyValues]
    value_stream_filters: List[HierarchyValues]
    geography_filters: List[HierarchyValues]

    groupings: List[RelatedEntity]

    department_permission_filters: List[HierarchyValues]
    value_stream_permission_filters: List[HierarchyValues]
    geography_permission_filters: List[HierarchyValues]

    enable_hierarchy_filter: bool
    enable_empty_summary_filter: bool
    summary_with_child: bool

    def filters_pair_for_grouping(
        self,
        related_entity: RelatedEntity,
    ) -> Tuple[List[HierarchyValues], List[HierarchyValues]]:
        if related_entity == RelatedEntity.department:
            return self.department_permission_filters, self.department_filters
        if related_entity == RelatedEntity.value_stream:
            return self.value_stream_permission_filters, self.value_stream_filters
        return self.geography_permission_filters, self.geography_filters

    @classmethod
    def from_default_args(
        cls,
        department_filters: Optional[List[HierarchyValues]] = None,
        value_stream_filters: Optional[List[HierarchyValues]] = None,
        geography_filters: Optional[List[HierarchyValues]] = None,
        groupings: Optional[List[RelatedEntity]] = None,
        department_permission_filters: Optional[List[HierarchyValues]] = None,
        value_stream_permission_filters: Optional[List[HierarchyValues]] = None,
        geography_permission_filters: Optional[List[HierarchyValues]] = None,
        enable_hierarchy_filter: bool = True,
        enable_empty_summary_filter: bool = True,
        summary_with_child: bool = True,
    ) -> 'QueryParams':
        return QueryParams(
            department_filters=department_filters or [],
            value_stream_filters=value_stream_filters or [],
            geography_filters=geography_filters or [],
            groupings=groupings or [],
            department_permission_filters=department_permission_filters or [],
            value_stream_permission_filters=value_stream_permission_filters or [],
            geography_permission_filters=geography_permission_filters or [],
            enable_empty_summary_filter=enable_empty_summary_filter,
            enable_hierarchy_filter=enable_hierarchy_filter,
            summary_with_child=summary_with_child,
        )

    @property
    def params_uses_department_table(self) -> bool:
        has_department_grouping = RelatedEntity.department in set(self.groupings)
        has_department_restrictions = bool(self.department_permission_filters)
        return has_department_grouping or bool(self.department_filters) or has_department_restrictions

    @property
    def params_uses_value_stream_table(self) -> bool:
        has_value_stream_grouping = RelatedEntity.value_stream in set(self.groupings)
        has_value_stream_restrictions = bool(self.value_stream_permission_filters)
        return has_value_stream_grouping or bool(self.value_stream_filters) or has_value_stream_restrictions

    @property
    def params_uses_geography_table(self) -> bool:
        has_geography_grouping = RelatedEntity.geography in set(self.groupings)
        has_geography_restrictions = bool(self.geography_permission_filters)
        return has_geography_grouping or bool(self.geography_filters) or has_geography_restrictions
