from typing import Any, Dict, Optional

import attr


@attr.s(auto_attribs=True)
class ChiefInfo:
    first_name: str
    last_name: str
    first_name_en: str
    last_name_en: str
    login: str


@attr.s(auto_attribs=True)
class GroupingInstanceInfo:
    name: str
    name_en: str
    id: int
    url: str
    instance_class: str
    chief: Optional[ChiefInfo]
    parent_id: Optional[int]

    @classmethod
    def default(cls) -> 'GroupingInstanceInfo':
        return GroupingInstanceInfo(
            name='',
            name_en='',
            id=0,
            url='',
            instance_class='',
            chief=None,
            parent_id=None,
        )

    @classmethod
    def from_dict(cls, values: Dict[str, Any], alias_name: str) -> 'GroupingInstanceInfo':
        return GroupingInstanceInfo(
            name=values[f'{alias_name}.name'],
            name_en=values[f'{alias_name}.name_en'],
            id=values[f'{alias_name}.id'],
            url=values[f'{alias_name}.url'],
            instance_class=values[f'{alias_name}.instance_class'],
            chief=None,
            parent_id=values[f'{alias_name}.parent_id'],
        )


@attr.s(auto_attribs=True)
class SummaryResults:
    headcount: int
    working: int
    offer: int
    balance: int
    vacancies: int
    vacancies_plan_new: int
    vacancies_plan_replacement: int
    credit: int
    working_crossing: int
    offer_crossing: int
    vacancies_plan_crossing: int
    vacancies_crossing: int

    def plus(self, result: 'SummaryResults') -> 'SummaryResults':
        return SummaryResults(
            headcount=self.headcount + result.headcount,
            working=self.working + result.working,
            offer=self.offer + result.offer,
            balance=self.balance + result.balance,
            vacancies=self.vacancies + result.vacancies,
            vacancies_plan_new=self.vacancies_plan_new + result.vacancies_plan_new,
            vacancies_plan_replacement=self.vacancies_plan_replacement + result.vacancies_plan_replacement,
            credit=self.credit + result.credit,
            working_crossing=self.working_crossing + result.working_crossing,
            offer_crossing=self.offer_crossing + result.offer_crossing,
            vacancies_plan_crossing=self.vacancies_plan_crossing + result.vacancies_plan_crossing,
            vacancies_crossing=self.vacancies_crossing + result.vacancies_crossing,
        )

    @classmethod
    def default(cls, headcount: int = 0) -> 'SummaryResults':
        return SummaryResults(
            balance=0,
            headcount=headcount,
            working=0,
            offer=0,
            vacancies=0,
            vacancies_plan_new=0,
            vacancies_plan_replacement=0,
            credit=0,
            working_crossing=0,
            offer_crossing=0,
            vacancies_plan_crossing=0,
            vacancies_crossing=0,
        )

    @classmethod
    def from_dict(cls, values: Dict[str, Any]) -> 'SummaryResults':
        return SummaryResults(
            headcount=values['headcount'],
            working=values['working'],
            offer=values['offer'],
            vacancies=values['vacancies'],
            vacancies_plan_new=values['vacancies_plan_new'],
            vacancies_plan_replacement=values['vacancies_plan_replacement'],
            credit=values['credit'],
            working_crossing=values['working_crossing'],
            offer_crossing=values['offer_crossing'],
            vacancies_plan_crossing=values['vacancies_plan_crossing'],
            vacancies_crossing=values['vacancies_crossing'],
            balance=values['balance'],
        )


@attr.s(auto_attribs=True)
class Result:
    summary: SummaryResults
    summary_without_children: SummaryResults
    next_level_grouping: Dict[int, 'Result']
    children: Dict[int, 'Result']
    id: Optional[int] = None
    current_level_grouping: Optional[str] = None
    grouping_instance_info: Optional[GroupingInstanceInfo] = None

    @classmethod
    def default(
        cls,
        result_id: Optional[int] = None,
        grouping: Optional[str] = None,
        grouping_instance_info: Optional[GroupingInstanceInfo] = None,
        children: Dict[int, 'Result'] = None,
        next_level_grouping: Dict[int, 'Result'] = None,
        summary: Optional[SummaryResults] = None,
        summary_without_children: Optional[SummaryResults] = None,
    ) -> 'Result':
        return Result(
            id=result_id,
            current_level_grouping=grouping,
            summary=summary or SummaryResults.default(),
            summary_without_children=summary_without_children or SummaryResults.default(),
            grouping_instance_info=grouping_instance_info,
            next_level_grouping=next_level_grouping or {},
            children=children or {},
        )
