from typing import Iterable, List

import attr

from django.utils.translation import get_language

from staff.lib import xlsx

from staff.headcounts.headcounts_summary.query_builder import RelatedEntity
from staff.headcounts.headcounts_summary.export_summary_model import SummaryModelRow, Name


@attr.s(auto_attribs=True)
class Field:
    name_en: str
    name_ru: str
    help_text: str
    data_field_name: str
    column: xlsx.Column

    @property
    def name(self) -> str:
        return self.name_en if get_language() == 'en' else self.name_ru


_without_outline_options = {'level': 0}
_within_outline_options = {'level': 1, 'hidden': True}
_collapsed_options = {'level': 0, 'collapsed': True}


STATIC_COLUMNS = (
    Field(
        'Limit', 'Потолок', '[1]',
        'total_hc',
        xlsx.Column(width=11, options=_without_outline_options),
    ),
    Field(
        'Actual HC', 'Численность работающих сотрудников', '[2]',
        'current_count',
        xlsx.Column(width=11, options=_without_outline_options),
    ),
    Field(
        'Offer', 'Офферы', '[3]',
        'offers',
        xlsx.Column(width=11, options=_without_outline_options),
    ),
    Field(
        'Balance', 'Баланс', '[1]-[2]-[3]=[4]=\n[5]+[6]+[7]-[8]',
        'current_balance',
        xlsx.Column(width=11, options=_without_outline_options),
    ),
    Field(
        'Opened vacancy', 'Открытые вакансии', '[5]',
        'vacancy_open',
        xlsx.Column(width=11, options=_without_outline_options),
    ),
    Field(
        'Unassigned headcount', 'Незанятый хедкаунт', '[6]',
        'vacancy_plan',
        xlsx.Column(width=11, options=_collapsed_options),
    ),
    Field(
        'Unassigned new headcount', 'Незанятый хедкаунт новый', '[6.1]',
        'vacancy_plan_new',
        xlsx.Column(width=11, options=_within_outline_options),
    ),
    Field(
        'Unassigned replacement headcount', 'Незанятый хедкаунт на замену уволенным', '[6.2]',
        'vacancy_plan_replacement',
        xlsx.Column(width=11, options=_within_outline_options),
    ),
    Field(
        'Credit', 'Кредит', '[7]',
        'reserve_new',
        xlsx.Column(width=11, options=_without_outline_options),
    ),
    Field(
        'Crossheadcount', 'Пересечения', '[8]',
        'crossheadcount',
        xlsx.Column(width=11, options=_collapsed_options),
    ),
    Field(
        'Сrossing with the working', 'Пересечения с работающими', '[8.1]',
        'replacement_to_working',
        xlsx.Column(width=11, options=_within_outline_options),
    ),
    Field(
        'Сrossing with the vacancies', 'Пересечения с офферами', '[8.2]',
        'offers_for_working_replacement',
        xlsx.Column(width=11, options=_within_outline_options),
    ),
    Field(
        'Сrossing with the offers', 'Пересечения с вакансиями', '[8.3]',
        'vacancy_open_for_working_replacement',
        xlsx.Column(width=11, options=_within_outline_options),
    ),
    Field(
        'Сrossing with the unassigned headcount', 'Пересечения с незанятым хедкаутом', '[8.4]',
        'vacancy_plan_for_working_replacement',
        xlsx.Column(width=11, options=_within_outline_options),
    ),
)


class SummarySheetPresenter(xlsx.SheetPresenter):
    outline_settings = {'symbols_below': 0, 'symbols_right': 0}
    freeze_panes = {'row': 2, 'col': 0}

    _heading_format = {
        'align': 'center',
        'valign': 'vcenter',
        'text_wrap': True,
        'bold': True,
        'bg_color': '#FFCC00',
    }

    _help_heading_format = {
        'align': 'center',
        'valign': 'vcenter',
        'text_wrap': True,
        'bold': True,
        'font_color': '#808080',
    }

    _gray_row_format = {
        'font_color': '#D3D3D3',
    }

    _model: Iterable[SummaryModelRow]

    def __init__(self, model: Iterable[SummaryModelRow], user_groupings: List[RelatedEntity], lang: str):
        super().__init__(model, 'Summary')
        self._user_groupings = user_groupings
        self._lang = lang

    def _select_lang(self, name: Name) -> str:
        if self._lang == 'en':
            return name.name_en
        return name.name

    def _grouping_to_text(self, grouping: RelatedEntity) -> str:
        names = {
            RelatedEntity.department: Name(name='Департамент', name_en='Department'),
            RelatedEntity.value_stream: Name(name='VS', name_en='VS'),
            RelatedEntity.geography: Name(name='География', name_en='Geo'),
        }
        return self._select_lang(names[grouping])

    def help_row(self) -> xlsx.ViewRow:
        view_cells = []

        for user_grouping in self._user_groupings:
            view_cells.append(xlsx.ViewCell(text=self._grouping_to_text(user_grouping)))
            view_cells.append(xlsx.ViewCell(text='Иерархия'))

        view_cells += [
            xlsx.ViewCell(field.help_text, self._help_heading_format)
            for field in STATIC_COLUMNS
        ]

        return xlsx.ViewRow(view_cells)

    def column_caption_row(self) -> xlsx.ViewRow:
        columns_captions_cells = []
        for _ in self._user_groupings:
            columns_captions_cells.append(xlsx.ViewCell(text=''))
            columns_captions_cells.append(xlsx.ViewCell(text=''))

        columns_captions_cells += [
            xlsx.ViewCell(field.name, self._heading_format)
            for field in STATIC_COLUMNS
        ]
        return xlsx.ViewRow(columns_captions_cells)

    def rows(self) -> Iterable[xlsx.ViewRow]:
        yield self.help_row()
        yield self.column_caption_row()

        for row in self._model:
            row_format = None if row.intranet_status else self._gray_row_format

            yield xlsx.ViewRow(
                cells=self._row_cells_from_model_row(row),
                row_format=row_format,
            )

    def _row_cells_from_model_row(self, row: SummaryModelRow) -> List[xlsx.ViewCell]:
        result = []

        for idx, _ in enumerate(self._user_groupings):
            if idx >= len(row.hierarchy_by_groupings):
                result.append(xlsx.ViewCell(text=''))
                result.append(xlsx.ViewCell(text=''))
            else:
                grouping = row.hierarchy_by_groupings[idx]
                result.append(xlsx.ViewCell(text=self._select_lang(grouping.hierarchy[-1])))
                hierarchy_items = [self._select_lang(item) for item in grouping.hierarchy]
                result.append(xlsx.ViewCell(text='→'.join(hierarchy_items)))

        result += [
            xlsx.ViewCell(text=str(row.headcount)),
            xlsx.ViewCell(text=str(row.working)),
            xlsx.ViewCell(text=str(row.offers)),
            xlsx.ViewCell(text=str(row.balance)),
            xlsx.ViewCell(text=str(row.vacancy)),
            xlsx.ViewCell(text=str(row.vacancy_plan)),
            xlsx.ViewCell(text=str(row.vacancy_plan_new)),
            xlsx.ViewCell(text=str(row.vacancy_plan_replacement)),
            xlsx.ViewCell(text=str(row.credit)),
            xlsx.ViewCell(text=str(row.crossing)),
            xlsx.ViewCell(text=str(row.working_crossing)),
            xlsx.ViewCell(text=str(row.offer_crossing)),
            xlsx.ViewCell(text=str(row.vacancy_crossing)),
            xlsx.ViewCell(text=str(row.vacancy_plan_crossing)),
        ]
        return result

    def columns(self) -> Iterable[xlsx.Column]:
        columns = []

        for _ in self._user_groupings:
            columns.append(xlsx.Column(width=11, options=_without_outline_options))
            columns.append(xlsx.Column(width=55, options=_without_outline_options))

        columns += [f.column for f in STATIC_COLUMNS]

        return columns
