from django.db import models

from staff.budget_position.models import BudgetPosition
from staff.departments.models import Department
from staff.lib.startrek.issues import startrek_issue_key_validator
from staff.person.models import Staff


class CreditManagementApplication(models.Model):
    """
    Model represents credit management application(form, заявка)
    """
    class Meta:
        permissions = (
            ('can_cancel_applications', 'Can cancel applications'),
            ('can_confirm_applications', 'Can confirm applications'),
        )

    author = models.ForeignKey(Staff, null=False)
    created_at = models.DateTimeField(auto_now_add=True)
    modified_at = models.DateTimeField(auto_now=True)
    closed_at = models.DateTimeField(null=True)
    comment = models.CharField(max_length=512, null=False, blank=False)
    startrek_headcount_key = models.CharField(
        null=True,
        blank=True,
        max_length=32,
        validators=[startrek_issue_key_validator],
    )
    is_active = models.BooleanField(default=True)


class CreditManagementApplicationRow(models.Model):
    """
    Model represents a row with 2 budget position from credit management application
    """
    credit_budget_position = models.ForeignKey(BudgetPosition, null=False)
    repayment_budget_position = models.ForeignKey(BudgetPosition, null=False, related_name='+')
    application = models.ForeignKey(CreditManagementApplication, null=False, related_name='rows')


class AllowedHeadcountOverdraft(models.Model):
    """
    Represents allowed overdraft for department
    """
    department = models.OneToOneField(Department)
    percents_with_child_departments = models.DecimalField(
        null=True,
        max_digits=4,
        decimal_places=2,
        verbose_name='Allowed overdraft in percents for department and its childs',
    )

    def __str__(self):
        return f'overdraft for {self.department.url}'
