from operator import and_
from typing import List, Dict, Iterable, Optional

from django.conf import settings
from django.db.models import Count
from django.db.models.query import Q, QuerySet, ValuesQuerySet, ValuesListQuerySet

from staff.budget_position.models import Reward
from staff.departments.models import Department, HeadcountPosition, Vacancy
from staff.headcounts.forms import QUANTITY_FILTER
from staff.headcounts.permissions import Permissions
from staff.lib.models.mptt import get_heirarchy_filter_query, filter_by_heirarchy
from staff.lib.models.query import build_search_query
from staff.oebs.constants import PERSON_POSITION_STATUS


# TODO: STAFF-17443: Remove in favour of BudgetPositionAssignmentFilterContext
class PositionsFilterContext:
    position_fields = (
        'id',
        'code',
        'name',

        'department_id',
        'department__id',
        'department__url',
        'department__name',
        'department__name_en',

        'valuestream_id',
        'valuestream__id',
        'valuestream__url',
        'valuestream__name',
        'valuestream__name_en',

        'geo',

        'headcount',
        'category_is_new',

        'current_person__login',
        'current_person__first_name',
        'current_person__last_name',
        'current_person__first_name_en',
        'current_person__last_name_en',
        'current_login',  # TODO: выпилить после STAFF-13508

        'status',
        'is_crossing',
        'main_assignment',
        'replacement_type',

        'previous_person__login',
        'previous_person__first_name',
        'previous_person__last_name',
        'previous_person__first_name_en',
        'previous_person__last_name_en',
        'previous_login',  # TODO: выпилить после STAFF-13508

        'index',
        'prev_index',
        'next_index',
        'reward_id',
    )

    dep_fields = (
        'id',
        'parent_id',
        'url',
        'tree_id',
        'lft',
        'rght',
        'name',
        'name_en',
        'level',
        'description',
        'description_en',
        'wiki_page',
        'maillists',
        'clubs',
        'intranet_status',
        'instance_class',
    )

    filter_id = None

    def __init__(self, observer_permissions: Permissions = None, exclude_reserve=True, **kwargs):
        self.observer_permissions = observer_permissions

        self.filter_q = Q()

        if kwargs.get('position_statuses'):
            self.filter_q &= Q(status__in=kwargs['position_statuses'])

        if kwargs.get('category_is_new') is not None:
            self.filter_q &= Q(category_is_new=kwargs['category_is_new'])

        if kwargs.get('is_crossing') is not None:
            self.filter_q &= Q(is_crossing=kwargs['is_crossing'])

        if kwargs.get('main_assignment') is not None:
            self.filter_q &= Q(main_assignment=kwargs['main_assignment'])

        if kwargs.get('replacement_type'):
            self.filter_q &= Q(replacement_type__in=kwargs['replacement_type'])

        if kwargs.get('quantity'):
            quantity = kwargs['quantity']
            if quantity == QUANTITY_FILTER.ZERO:
                self.filter_q &= Q(headcount=0)
            elif quantity == QUANTITY_FILTER.GREATER_THAN_ZERO:
                self.filter_q &= Q(headcount__gt=0)
            elif quantity == QUANTITY_FILTER.LESS_THAN_ZERO:
                self.filter_q &= Q(headcount__lt=0)

        if kwargs.get('current_person'):
            self.filter_q &= Q(current_person=kwargs['current_person'])

        if kwargs.get('replaced_person'):
            self.filter_q &= Q(previous_person=kwargs['replaced_person'])

        if kwargs.get('code') is not None:
            self.filter_q &= Q(code=kwargs['code'])

        if kwargs.get('codes'):
            self.filter_q &= Q(code__in=kwargs['codes'])

        if kwargs.get('department_url'):
            self.filter_q &= Q(department__url=kwargs['department_url'])

        if kwargs.get('department'):
            self.filter_q &= get_heirarchy_filter_query(
                mptt_objects=[kwargs['department']],
                by_children=True,
                filter_prefix='department__',
                include_self=True,
            )

        if kwargs.get('value_stream'):
            self.filter_q &= get_heirarchy_filter_query(
                mptt_objects=[kwargs['value_stream']],
                by_children=True,
                filter_prefix='valuestream__',
                include_self=True,
            )

        if kwargs.get('geography'):
            self.filter_q &= self._get_geography_filter(kwargs.get('geography'))

        if kwargs.get('search_text'):
            text = kwargs['search_text']
            self.filter_q &= self._get_qs_for_searching_by_all_details(text)

        if kwargs.get('category'):
            self.filter_q &= self._get_category_filter(kwargs['category'])

        if exclude_reserve:
            self.filter_q &= ~(Q(status=PERSON_POSITION_STATUS.RESERVE) & Q(headcount__gte=0))

    def _get_category_filter(self, categories: List[str]) -> Q:
        reward_qs = (
            Reward.objects
            .filter(oebs_instance__category__in=categories)
            .values_list('oebs_instance__scheme_id', flat=True)
        )
        return Q(reward_id__in=reward_qs)

    def _get_qs_for_searching_by_all_details(self, text: str) -> Q:
        occupied_status = Q(status=PERSON_POSITION_STATUS.OCCUPIED)
        offer_status = Q(status=PERSON_POSITION_STATUS.OFFER)
        vacancy_status = Q(status=PERSON_POSITION_STATUS.VACANCY_OPEN)
        free_status = Q(status__in=(PERSON_POSITION_STATUS.VACANCY_PLAN, PERSON_POSITION_STATUS.MATERNITY))

        person_fields = [
            'current_person__login',
            'current_person__first_name',
            'current_person__last_name',
            'current_person__first_name_en',
            'current_person__last_name_en',
        ]
        person_query = build_search_query(text, person_fields, and_)

        vacancies_qs = self._get_base_vacancies_qs()

        offer_fields = ['candidate_first_name', 'candidate_last_name']
        offer_query = vacancies_qs.filter(build_search_query(text, offer_fields, and_))

        vacancy_query = vacancies_qs.filter(name__icontains=text)
        ticket_query = vacancies_qs.filter(self._ticket_q(text))

        return (
            person_query & occupied_status
            | Q(name__icontains=text) & (free_status | (offer_status | vacancy_status) & ~Q(code__in=vacancies_qs))
            | Q(code__in=offer_query) & offer_status
            | Q(code__in=vacancy_query) & vacancy_status
            | Q(code__in=ticket_query) & (offer_status | vacancy_status)
        )

    def _ticket_q(self, search_text: str) -> Q:
        ticket_query = Q()
        for part in search_text.split():
            ticket_query |= Q(ticket__icontains=part)

        return ticket_query

    def _get_base_vacancies_qs(self) -> ValuesListQuerySet:
        vacancies_qs = (
            Vacancy.objects
            .filter(is_active=True, headcount_position_code__isnull=False)
            .values_list('headcount_position_code')
        )
        return vacancies_qs

    def departments_qs(self, fields: Optional[Iterable[str]] = None) -> ValuesQuerySet:
        if fields is None:
            fields = self.dep_fields

        return (
            Department.all_types
            .annotate(Count('positions'))
            .filter(Q(intranet_status=1) | Q(positions__count__gt=0))
            .exclude(id=settings.INTRANET_DISMISSED_STAFF_DEPARTMENT_ID)
            .values(*fields)
        )

    def positions_objects_qs(self) -> QuerySet:
        qs = (
            self.observer_permissions.available_headcounts()
            if self.observer_permissions
            else HeadcountPosition.objects.all()
        )
        return qs.filter(self.filter_q)

    def positions_qs(self, fields: Iterable[str] = None) -> ValuesQuerySet:
        fields = fields or self.position_fields
        qs = self.positions_objects_qs().values(*fields)
        return qs

    def positions_quantity_qs(self, entity_dep_field_name) -> ValuesQuerySet:
        positions_qs = self.positions_qs()

        qs = (
            positions_qs
            .values(entity_dep_field_name)
            .annotate(qty=Count('code'))
        )

        return qs

    def positions_for_links(self, or_filters: List[Dict]) -> List[Dict]:
        if not or_filters:
            return []

        filter_qs = Q(id=None)
        for filter_ in or_filters:
            filter_qs |= Q(**filter_)

        assert self.observer_permissions
        by_department_filter_q = Q(
            department_id__in=self.observer_permissions.filter_by_observer(Department.objects.all())
        )

        by_valuestream_filter_q = Q(
            valuestream_id__in=self.observer_permissions.filter_by_observer(Department.valuestreams.all())
        )

        qs = (
            HeadcountPosition.objects
            .filter(filter_qs)
            .filter(by_department_filter_q | by_valuestream_filter_q)
            .exclude(status=PERSON_POSITION_STATUS.RESERVE)
            .values(*self.position_fields)
        )

        return list(qs)

    @staticmethod
    def _get_geography_filter(geography):
        oebs_codes = filter_by_heirarchy(
            query_set=Department.geography.all(),
            mptt_objects=[geography],
            by_children=True,
            include_self=True,
        ).values('geography_instance__oebs_code')

        return Q(geo__in=oebs_codes)
