from typing import Tuple, Optional

from django.db.models.aggregates import Max
from django.http import JsonResponse
from django.db.models import QuerySet
from django.views.decorators.csrf import csrf_exempt
from django.views.decorators.http import require_http_methods

from staff.departments.models import HeadcountPosition, Vacancy
from staff.lib.utils.qs_values import repack_related
from staff.lib.forms.errors import sform_general_error
from staff.oebs.constants import PERSON_POSITION_STATUS
from staff.person.models import Staff, StaffExtraFields
from staff.headcounts.forms import MultiplePersonsForm


PAGE_SIZE = 1000


def _staff_value_stream_url(valuestream_url: str) -> str:
    assert valuestream_url
    return (
        valuestream_url
        if valuestream_url.startswith('svc_')
        else 'svc_' + valuestream_url
    )


def _staff_by_valuestream_url_qs(valuestream_url: str) -> QuerySet:
    fields = (
        'login',
        'last_name',
        'first_name',
        'last_name_en',
        'first_name_en',
        'department__id',
        'department__url',
        'department__name',
        'department__name_en',
    )

    hp_qs = (
        HeadcountPosition.objects
        .filter(status=PERSON_POSITION_STATUS.OCCUPIED, valuestream__url=valuestream_url)
        .values_list('current_person_id')
    )

    qs = Staff.objects.filter(id__in=hp_qs).values(*fields)
    return qs


def _headcount_positions_by_valuestream_url_qs(
    valuestream_url: str,
    continuation_token: str,
    page_size: int,
) -> Tuple[list, Optional[str]]:
    hp_qs = (
        HeadcountPosition.objects
        .filter(
            headcount=1,
            status__in=(
                PERSON_POSITION_STATUS.OCCUPIED,
                PERSON_POSITION_STATUS.VACANCY_OPEN,
                PERSON_POSITION_STATUS.OFFER,
            ),
            valuestream__url=valuestream_url,
        )
    )

    if continuation_token:
        hp_qs = hp_qs.filter(code__gt=continuation_token)

    hp_qs = (
        hp_qs
        .values('code')
        .annotate(has_person=Max('current_person_id'))
        .order_by('code')
    )[:page_size]

    headcount_positions = list(hp_qs)

    result_continuation_token: Optional[str] = None

    if len(headcount_positions) >= page_size:
        result_continuation_token = headcount_positions[-1]['code']

    return headcount_positions, result_continuation_token


def _staff_by_headcount_positions_qs(headcount_positions: list) -> QuerySet:
    fields = (
        'login',
        'last_name',
        'first_name',
        'last_name_en',
        'first_name_en',
        'department__id',
        'department__url',
        'department__name',
        'department__name_en',
    )

    position_codes = [position['code'] for position in headcount_positions]
    return Staff.objects.filter(budget_position__code__in=position_codes).values(*fields)


def _vacancies_by_headcount_positions_qs(headcount_positions: list) -> QuerySet:
    fields = (
        'id',
        'budget_position__code',
        'is_published',
        'name',
        'occupation__name',
        'occupation__description',
        'occupation__description_en',
        'department__id',
        'department__url',
        'department__name',
        'department__name_en',
    )

    position_codes = [position['code'] for position in headcount_positions if position['has_person'] is None]
    qs = Vacancy.objects.exclude(is_hidden=True)
    qs = qs.filter(budget_position__code__in=position_codes).values(*fields)

    return qs


@require_http_methods(['GET'])
def persons_by_vs(request):
    # пока без всякой паджинации, кажется мы можем отсюда эффективно отдавать до пары десятков тысяч
    # посмотрим лет через пять
    valuestream_url: str = request.GET.get('vs', '').strip()

    if not valuestream_url:
        return JsonResponse(status=400, data={'err': 'missing_vs_url'})

    valuestream_url = _staff_value_stream_url(valuestream_url)

    # но пока ограничимся 30к, это точно работает за приемлемое время
    result = [
        repack_related(person)
        for person in _staff_by_valuestream_url_qs(valuestream_url)[:30000]
    ]

    return JsonResponse(data={'persons': result})


@csrf_exempt
@require_http_methods(['POST'])
def vs_by_persons(request):
    form = MultiplePersonsForm(dict(request.POST))

    if not form.is_valid():
        return JsonResponse(data=form.errors, status=400)

    persons = form.cleaned_data['persons']

    if len(persons) > 500:
        return JsonResponse(data=sform_general_error('too_many_persons'), status=400)

    fields = (
        'current_login',
        'valuestream__url',
        'hr_product_id',
        'hr_product__service_id',
    )

    qs = HeadcountPosition.objects.filter(current_person__in=persons).values(*fields)

    result = {}
    for hp in qs:
        result[hp['current_login']] = {
            'vs_url': hp['valuestream__url'],
            'hr_product_id': hp['hr_product_id'],
            'abc_service_id': hp['hr_product__service_id'],
        }

    return JsonResponse(data=result)


@require_http_methods(['GET'])
def positions_by_vs(request):
    continuation_token: str = request.GET.get('continuation_token', None)
    valuestream_url: str = request.GET.get('vs', '').strip()

    if not valuestream_url:
        return JsonResponse(status=400, data={'err': 'missing_vs_url'})

    valuestream_url = _staff_value_stream_url(valuestream_url)
    headcount_positions, continuation_token = _headcount_positions_by_valuestream_url_qs(
        valuestream_url,
        continuation_token,
        PAGE_SIZE,
    )

    result = []
    for person in _staff_by_headcount_positions_qs(headcount_positions):
        person = repack_related(person)
        person['entity_type'] = 'person'
        result.append(person)

    for vacancy in _vacancies_by_headcount_positions_qs(headcount_positions):
        vacancy = repack_related(vacancy)
        vacancy['entity_type'] = 'vacancy'
        if not vacancy['is_published']:
            del vacancy['name']
        result.append(vacancy)

    data = {'positions': result}

    if continuation_token:
        data['continuation_token'] = continuation_token

    return JsonResponse(data=data)


@csrf_exempt
@require_http_methods(['POST'])
def occupation_by_logins(request):
    form = MultiplePersonsForm(dict(request.POST))

    if not form.is_valid():
        return JsonResponse(data=form.errors, status=400)

    persons = form.cleaned_data['persons']

    if len(persons) > 500:
        return JsonResponse(data=sform_general_error('too_many_persons'), status=400)

    result = dict(
        StaffExtraFields.objects
        .filter(staff__in=persons)
        .values_list('staff__login', 'occupation')
    )

    return JsonResponse(data=result)
