import json

from django.conf import settings
from django.template import loader
from django.views.decorators.http import require_http_methods
from django.views.decorators.csrf import ensure_csrf_cookie

from staff.budget_position.models import BudgetPosition
from staff.budget_position.workflow_service import (
    WorkflowRegistryService,
    ConflictWorkflowExistsError,
    MoveToMaternityData,
    OEBSError,
)
from staff.lib.decorators import responding_json
from staff.lib.forms.errors import sform_general_error
from staff.lib.startrek.issues import create_issue
from staff.person.models import Staff

from staff.headcounts.permissions import Permissions
from staff.headcounts.forms import MoveToMaternityForm


def create_ticket(budget_position: BudgetPosition, person: Staff, author: Staff):
    context = {
        'bp_code': budget_position.code,
        'login': person.login,
        'first_name_en': person.first_name_en,
        'last_name_en': person.last_name_en,
        'first_name': person.first_name,
        'last_name': person.last_name,
    }

    description = loader.render_to_string(
        template_name='startrek/maternity.html',
        context=context,
    )

    issue = create_issue(
        queue=settings.PROPOSAL_QUEUE_ID,
        createdBy=author.login,
        summary='Перевод в декрет',
        description=description,
        unique=f'maternity_{budget_position.code}_{person.login}_{settings.STAFF_HOST}',
        # TODO: change code to something else
    )

    return issue.key


@ensure_csrf_cookie
@responding_json
@require_http_methods(['POST', 'GET'])
def move_to_maternity(request):
    author = request.user.get_profile()

    if not Permissions(author).has_special_role():
        return {}, 403

    if request.method == 'GET':
        return MoveToMaternityForm().as_dict()

    form = MoveToMaternityForm(json.loads(request.body))

    if not form.is_valid():
        return form.errors_as_dict(), 400

    bp = form.cleaned_data['budget_position']
    person = form.cleaned_data['person']

    try:
        ticket_key = create_ticket(bp, person, author)
    except Exception:
        return sform_general_error(err_code='cant_create_ticket'), 500

    try:
        data = MoveToMaternityData(
            login=person.login,
            department_id=person.department_id,
            responsible_id=author.id,
            budget_position=bp.code,
            ticket=ticket_key,
        )
        workflow_id = WorkflowRegistryService().try_create_workflow_for_maternity(data)
    except ConflictWorkflowExistsError as e:
        for workflow in e.conflicting_workflows:
            if workflow.proposal_id:
                return sform_general_error(err_code='bp_conflict', params=workflow.proposal_id), 400

        return sform_general_error(err_code='bp_conflict'), 400
    except OEBSError as e:
        return sform_general_error(err_code='oebs_error', params=str(e)), 500

    return {'ticket_key': ticket_key, 'workflow_id': str(workflow_id)}, 200
