from django_idm_api.exceptions import RoleNotFound

from staff.departments.models import Department, DepartmentStaff, DepartmentRole, DepartmentRoles

from staff.idm.base import BaseActions, filter_q_factory


get_role_objects_filter_q = filter_q_factory({'manage_by_idm': True})


def get_role_objects_qs():
    return DepartmentRole.objects.filter(get_role_objects_filter_q())


class DepartmentRoleActions(BaseActions):
    @staticmethod
    def get_objects(fields):
        objects = get_role_objects_qs().values(*fields)
        return objects

    def set_value(self, values, key, value, obj=None):
        if key == DepartmentRoles.HR_PARTNER:
            value['fields'] = [{
                'slug': 'fincab_required',
                'name': {
                    'ru': 'Требуется финансовый кабинет',
                    'en': 'Financial Cabinet is Required',
                },
                'type': 'booleanfield',
            }]
        values[key] = value


get_objects_filter_q = filter_q_factory({'intranet_status': 1})


def get_objects_qs():
    return Department.objects.filter(get_objects_filter_q())


class DepartmentActions(BaseActions):
    role_model_cls = DepartmentStaff

    @staticmethod
    def get_objects(fields):
        objects = get_objects_qs().values(*fields)
        return objects

    @staticmethod
    def _get_object(object_id):
        try:
            return get_objects_qs().get(url=object_id)
        except Department.DoesNotExist:
            raise RoleNotFound('Role "department:%s" not found' % object_id)

    @staticmethod
    def _check_role_object(object_id):
        if not get_role_objects_qs().filter(id=object_id).exists():
            raise RoleNotFound('Role "department_role:%s" not found' % object_id)

    def _get_role_kwargs(self, person_id, role):
        department = self._get_object(role['department'])
        self._check_role_object(role['role'])
        return dict(
            department=department,
            role_id=role['role'],
            staff_id=int(person_id),
        )

    def set_value(self, values, key, value, obj=None):
        crumbs = [v for k, v in self.key_crumbs]
        crumbs.append(obj['url'])
        value['unique_id'] = '__'.join(crumbs)
        values[key] = value

    def add_role(self, person_id, role):
        role_kwargs = self._get_role_kwargs(person_id, role)
        created = self.role_model_cls.objects.get_or_create(**role_kwargs)[1]
        return created

    def remove_role(self, person_id, role):
        role_kwargs = self._get_role_kwargs(person_id, role)
        to_delete = self.role_model_cls.objects.filter(**role_kwargs)
        for obj in to_delete:
            obj.delete()

    def get_all_roles(self):
        gms = (
            self.role_model_cls.objects
            .filter(department__intranet_status=1)
            .filter(get_objects_filter_q(prefix='department'))
            .filter(get_role_objects_filter_q(prefix='role'))
            .values_list('staff__login', 'department__url', 'role_id')
        )

        for person_login, department_url, role_id in gms:
            key_crumbs = dict(self.key_crumbs)
            key_crumbs['department'] = str(department_url)
            key_crumbs['role'] = str(role_id)
            yield person_login, key_crumbs
