from copy import deepcopy
from collections import defaultdict

from django_idm_api.hooks import BaseHooks
from django_idm_api.exceptions import UserNotFound

from staff.person.models import Staff

from staff.idm.base import BaseActions
from staff.idm.group_actions import GroupActions, GroupMemberActions, GroupResponsibleActions
from staff.idm.robot_actions import RobotActions, RobotRoleActions
from staff.idm.user_attr_actions import UserAttributeActionsSuperuser, UserAttributeActionsStaff
from staff.idm.department_role_actions import DepartmentActions, DepartmentRoleActions


ROLES_TREE = {
    'code': 0,
    'roles': {
        'slug': 'type',
        'name': {
            'ru': 'Тип',
            'en': 'Type',
        },
        'values': {
            'groups': {
                '_actions_cls': GroupActions,
                'name': {
                    'ru': 'Управление группами',
                    'en': 'Manage Groups',
                },
                'roles': {
                    'slug': 'group',
                    'name': {
                        'ru': 'Группа',
                        'en': 'Group',
                    },
                    'values': {
                        '__id__': {
                            'name': {
                                'ru': '__name__',
                                'en': '__name__',
                            },
                            'roles': {
                                'slug': 'role',
                                'name': {
                                    'ru': 'Роль',
                                    'en': 'Role',
                                },
                                'values': {
                                    'member': {
                                        '_actions_cls': GroupMemberActions,
                                        'name': {
                                            'ru': 'Участник',
                                            'en': 'Member',
                                        }
                                    },
                                    'responsible': {
                                        '_actions_cls': GroupResponsibleActions,
                                        'name': {
                                            'ru': 'Ответственный',
                                            'en': 'Responsible',
                                        }
                                    }
                                }
                            }
                        }
                    }
                }
            },
            'robots': {
                '_actions_cls': RobotActions,
                'name': {
                    'ru': 'Управление роботами',
                    'en': 'Manage Robots',
                },
                'roles': {
                    'slug': 'robot',
                    'name': {
                        'ru': 'Робот',
                        'en': 'Robot',
                    },
                    'values': {
                        '__id__': {
                            'name': {
                                'ru': '__login__',
                                'en': '__login__',
                            },
                            'roles': {
                                'slug': 'role',
                                'name': {
                                    'ru': 'Роль',
                                    'en': 'Role',
                                },
                                'values': {
                                    'owner': {
                                        '_actions_cls': RobotRoleActions,
                                        'name': {
                                            'ru': 'Владелец',
                                            'en': 'Owner',
                                        }
                                    },
                                    'user': {
                                        '_actions_cls': RobotRoleActions,
                                        'name': {
                                            'ru': 'Пользователь',
                                            'en': 'User',
                                        }
                                    }
                                }
                            }
                        }
                    }
                }
            },
            'user_attrs': {
                'name': {
                    'ru': 'Управление свойствами пользователей',
                    'en': 'Manage Users Attributes',
                },
                'roles': {
                    'slug': 'user_attr',
                    'name': {
                        'ru': 'Атрибут пользователя',
                        'en': 'User Attribute',
                    },
                    'values': {
                        'is_superuser': {
                            '_actions_cls': UserAttributeActionsSuperuser,
                            'name': {
                                'ru': 'Superuser',
                                'en': 'Superuser',
                            },
                        },
                        'is_staff': {
                            '_actions_cls': UserAttributeActionsStaff,
                            'name': {
                                'ru': 'StaffAdmin',
                                'en': 'StaffAdmin',
                            },
                        }
                    }
                }
            },
            'department_roles': {
                '_actions_cls': DepartmentRoleActions,
                'name': {
                    'ru': 'Роли в подразделениях',
                    'en': 'Roles in Departments',
                },
                'roles': {
                    'slug': 'role',
                    'name': {
                        'ru': 'Роль',
                        'en': 'Role',
                    },
                    'values': {
                        '__id__': {
                            '_actions_cls': DepartmentActions,
                            'name': {
                                'ru': '__name__',
                                'en': '__name_en__',
                            },
                            'roles': {
                                'slug': 'department',
                                'name': {
                                    'ru': 'Подразделение',
                                    'en': 'Department',
                                },
                                'values': {
                                    '__url__': {
                                        'name': {
                                            'ru': '__name__',
                                            'en': '__name_en__',
                                        }
                                    }
                                }
                            }
                        }
                    }
                }
            },
        }
    }
}


class Hooks(BaseHooks):

    def info(self):
        """Вернуть полное дерево ролей"""
        actions = self._get_actions()
        info = actions.get_info()
        return info

    def add_role_impl(self, login, role, fields, **kwargs):
        """Добавить человеку роль"""
        person_id = self._get_person_id(login)
        actions = self._get_actions()
        actions.add_role(person_id=person_id, role=role)

    def remove_role_impl(self, login, role, data, is_fired, **kwargs):
        """Удалить роль человека"""
        person_id = self._get_person_id(login)
        actions = self._get_actions()
        actions.remove_role(person_id=person_id, role=role)

    def get_all_roles_impl(self, **kwargs):
        """Перечислить всех людей и их роли"""
        result = defaultdict(list)
        actions = self._get_actions()
        for login, role in actions.get_all_roles():
            result[login].append(role)
        return list(result.items())

    @staticmethod
    def _get_actions():
        roles_tree = deepcopy(ROLES_TREE)
        actions = BaseActions(tree_node=roles_tree, key_crumbs=[])
        return actions

    @staticmethod
    def _get_person_id(login):
        try:
            person_id = Staff.objects.values_list('id', flat=True).get(login=login)
        except Staff.DoesNotExist:
            raise UserNotFound('Requested user "%s" does not exist' % login)
        return person_id
