import csv

from django import forms
from django.contrib import admin
from django.http import HttpResponse

from staff.innerhire.models import Profession
from .models import Vacancy, Candidate
from staff.person.models import Staff
from staff.departments.models import Department


class LongCharField(forms.CharField):
    def widget_attrs(self, widget):
        return {'size': '88'}


class ProfessionAdmin(admin.ModelAdmin):

    list_display = ('name', 'name_en', 'is_active')
    actions = ['delete_selected', 'restore_selected']

    def delete_selected(self, request, queryset):
        queryset.update(is_active=False)

    def restore_selected(self, request, queryset):
        queryset.update(is_active=True)

    delete_selected.short_description = 'Удалить выбранные профессии'  # type: ignore
    restore_selected.short_description = 'Восстановить выбранные профессии'  # type: ignore


class VacancyAdminForm(forms.ModelForm):
    department = LongCharField(label='URL подразделения')
    recruiter = forms.CharField(label='Логин рекрутера')

    class Meta:
        fields = '__all__'
        model = Vacancy

    def __init__(self, *args, **kwargs):
        initial = kwargs.get('initial', {})
        instance = kwargs.get('instance', None)

        if instance:
            initial['department'] = instance.department.url
            initial['recruiter'] = instance.recruiter.login

        kwargs['initial'] = initial

        super(VacancyAdminForm, self).__init__(*args, **kwargs)

    def clean_recruiter(self):
        try:
            return (
                Staff.objects
                .filter(is_dismissed=False)
                .get(login=self.cleaned_data['recruiter'])
            )
        except Staff.DoesNotExist:
            raise forms.ValidationError(
                'No person with login `%s`' % self.cleaned_data['recruiter']
            )

    def clean_department(self):
        try:
            return (
                Department.objects
                .filter(intranet_status=1)
                .get(url=self.cleaned_data['department'])
            )
        except Department.DoesNotExist:
            raise forms.ValidationError(
                'No department with url `%s`' % self.cleaned_data['department']
            )


class VacancyAdmin(admin.ModelAdmin):
    list_display = (
        'name',
        'department',
        'description',
        'recruiter',
        'date_open',
        'date_close',
        'reason',
        'femida_link',
    )
    search_fields = (
        'department__name',
        'department__url',
        'name',
        'description',
        'recruiter__first_name',
        'recruiter__first_name_en',
        'recruiter__last_name',
        'recruiter__last_name_en',
        'recruiter__login',
        'femida_link',
    )
    list_filter = (
        'reason',
        'recommendation_program',
        'grade_program',
    )
    readonly_fields = ('grade_program', 'recommendation_program')

    actions = ['export_as_csv']

    def export_as_csv(self, request, queryset):
        response = HttpResponse(content_type='text/csv')
        response['Content-Disposition'] = 'attachment; filename=femida_inner_export.csv'
        field_names = ['femida_link', 'name', 'description']
        writer = csv.writer(response)
        writer.writerow(field_names)
        for obj in Vacancy.objects.all():
            writer.writerow([str(getattr(obj, field)).encode('utf-8') for field in field_names])
        return response

    export_as_csv.short_description = "Export"  # type: ignore

    form = VacancyAdminForm


admin.site.register(Profession, ProfessionAdmin)
admin.site.register(Vacancy, VacancyAdmin)
admin.site.register(Candidate)
