from datetime import date

from django.utils.translation import get_language
from django.utils.html import escape

from staff.lib.models.base import get_i_field
from staff.lib.models.departments_chain import get_departments_tree
from staff.lib.models.mptt import filter_by_heirarchy
from staff.lib.utils.qs_values import localize, extract_related
from staff.departments.models import DepartmentStaff, DepartmentRoles
from .models import (
    Candidate,
    CANDIDATE_INFO_MAX_LEN,
)


def filter_by_departments_heirarchy(vacancies_qs, departments):
    return filter_by_heirarchy(
        vacancies_qs,
        departments,
        by_children=True,
        include_self=True,
        filter_prefix='department__',
    )


def filter_by_chief_departments_heirarchy(vacancies_qs, chief):
    roles_qs = (
        DepartmentStaff.objects
        .filter(role_id=DepartmentRoles.CHIEF.value, staff=chief)
        .values(
            'department__tree_id',
            'department__lft',
            'department__rght',
        )
    )

    if not roles_qs:
        return vacancies_qs.none()

    return filter_by_heirarchy(
        vacancies_qs,
        roles_qs,
        by_children=True,
        prefix='department__',
        filter_prefix='department__',
        include_self=True,
    )


class DepartmentsTree(object):

    def __init__(self, departments):
        self.departments_tree = get_departments_tree(
            departments=departments,
            fields=['native_lang', 'name', 'name_en', 'url', 'id'],
        )

    def __getitem__(self, department_id):
        department_chain = self.departments_tree[department_id]
        return [
            {'name': get_i_field(d, 'name'), 'url': d['url'], 'id': d['id']}
            for d in department_chain
        ]


def get_direct_chief(department_chain_ids):
    roles_qs = (
        DepartmentStaff.objects
        .filter(
            role_id=DepartmentRoles.CHIEF.value,
            department__in=department_chain_ids,
        )
        .values(
            'staff__native_lang',
            'staff__first_name',
            'staff__last_name',
            'staff__first_name_en',
            'staff__last_name_en',
            'staff__login',
            'staff',
        )
        .order_by('-department__level')
    )
    chief = roles_qs[0]

    return {
        'first_name': get_i_field(chief, 'first_name', 'staff__'),
        'last_name': get_i_field(chief, 'last_name', 'staff__'),
        'login': chief['staff__login'],
        'id': chief['staff']
    }


def warp_recruiter(vacancy):
    vacancy['recruiter'] = {
        'first_name': get_i_field(vacancy, 'first_name', 'recruiter__'),
        'last_name': get_i_field(vacancy, 'last_name', 'recruiter__'),
        'login': vacancy.pop('recruiter__login'),
        'id': vacancy.pop('recruiter')
    }
    del vacancy['recruiter__native_lang']
    del vacancy['recruiter__first_name']
    del vacancy['recruiter__last_name']
    del vacancy['recruiter__first_name_en']
    del vacancy['recruiter__last_name_en']


def localize_vacancy_attr(vacancy, attr):
    attr_en = attr + '_en'
    if attr in vacancy:
        if 'ru' not in get_language() and vacancy[attr_en]:
            vacancy[attr] = vacancy[attr_en]
        del vacancy[attr_en]

        return vacancy[attr]


def localize_vacancy(vacancy):
    for attr in ('name', 'description', 'profession__name'):
        localize_vacancy_attr(vacancy, attr)

    return vacancy


def get_candidates():
    candidates_qs = (
        Candidate.objects
        .filter(date_close=None)
        .filter(person__is_dismissed=False)
        .values(
            'description_source',
            'date_open',
            'person__id',
            'person__login',
            'person__first_name',
            'person__first_name_en',
            'person__last_name',
            'person__last_name_en',
            'person__position',
            'person__position_en',
            'person__department_id',
            'person__is_big_boss',
            'person__office__id',
            'person__office__name',
            'person__office__name_en',
        )
        .order_by('-date_open')
    )

    departments_ids = set()
    candidates = []
    for candidate in candidates_qs:
        candidate = localize(candidate)
        candidate['description'] = escape(candidate.pop('description_source'))
        candidate.update(extract_related(candidate, 'person'))
        candidate['office'] = extract_related(candidate, 'office')
        departments_ids.add(candidate['department_id'])
        candidates.append(candidate)

    departments_tree = DepartmentsTree(departments_ids)

    for candidate in candidates:
        candidate['department'] = departments_tree[candidate.pop('department_id')]

    return candidates


def get_candidate_info(person):
    try:
        return Candidate.objects.filter(date_close=None, person=person)[0]
    except IndexError:
        return None


class CandidateInfoIsTooLong(Exception):
    pass


def update_candidate_info(person, info, author):
    info_source = info.rstrip()

    if len(info_source) > CANDIDATE_INFO_MAX_LEN:
        raise CandidateInfoIsTooLong

    candidate_info = get_candidate_info(person)

    if candidate_info:
        if candidate_info.description_source == info_source:
            return candidate_info

        candidate_info.date_close = date.today()
        candidate_info.save()

    if info_source:
        return Candidate.objects.create(
            person=person,
            description_source=info_source,
            description=info_source,
            author=author,
        )
