from django import forms

from staff.lib.forms.fields import StaffField

from staff.person_profile.controllers.gpg_key import (
    gpg_fingerprint,
    GPGKeyError,
)

from staff.person_profile.controllers.ssh_key import ssh_fingerprint, SSHKeyError

from staff.lib.forms.objects import FormGrid

from .models import GPGKey, BannedSSHKey

jquery_url = '//js.static.yandex.net/jquery/1.3.2/_jquery.js'


class GPGKeyForm(forms.Form):
    """Форма валидации GPG ключа"""
    fingerprint = forms.CharField(required=False)
    key = forms.CharField(required=False)
    description = forms.CharField(required=False)

    def clean_key(self):
        value = self.cleaned_data.get('key', '').strip()
        try:
            self.cleaned_data['fingerprint'] = gpg_fingerprint(value)
        except GPGKeyError:
            # код сроком жизни в неделю (отсчёт с 17.06.2015)
            self._errors['key'] = ['Invalid GPG key']

        return value


class GPGKeyFormGrid(FormGrid):
    form_cls = GPGKeyForm


class GPGKeyAdminForm(forms.ModelForm):
    staff = StaffField(required=True, label='Владелец ключа')
    description = forms.CharField(max_length=500, label='Описание')
    fingerprint = forms.CharField(max_length=100, label='Фингерпринт')

    class Meta:
        model = GPGKey
        fields = '__all__'

    class Media:
        js = (jquery_url,)


class BannedSSHAdminForm(forms.ModelForm):
    description = forms.CharField(max_length=255)
    key = forms.CharField(widget=forms.Textarea, required=True)
    fingerprint_sha256 = forms.CharField(widget=forms.TextInput(attrs={'readonly': 'readonly'}), required=False)

    def clean(self):
        cleaned_data = super(BannedSSHAdminForm, self).clean()
        key = cleaned_data.get('key', '').strip()

        try:
            fingerprint = ssh_fingerprint(key, 'sha256')
        except SSHKeyError:
            raise forms.ValidationError('Invalid SSH key')

        cleaned_data['fingerprint_sha256'] = fingerprint
        cleaned_data['key'] = key
        return cleaned_data

    class Meta:
        model = BannedSSHKey
        fields = '__all__'
