from datetime import datetime
import logging

from django.conf import settings

from staff.celery_app import app
from staff.keys.models import SSHKey
from staff.lib import requests
from staff.lib.db import atomic
from staff.lib.tasks import LockedTask


logger = logging.getLogger(__name__)
LIMIT = 100


def to_datetime(d):
    return datetime.strptime(d, '%Y-%m-%dT%H:%M:%S')


def get_keys_from_cauth(date_from):
    has_more = True
    params = {
        'date_from': date_from.strftime('%Y-%m-%dT%H:%M:%S'),
        'limit': LIMIT,
    }
    while has_more:
        resp = requests.get(settings.CAUTH_PUBLIC_HOST + '/public_keys/', params=params)
        if resp.status_code != 200:
            logger.warning('Cauth didn\'t return keys with code {} due to:\n{}'.format(
                resp.status_code,
                resp.content,
            ))
            return
        keys = resp.json()['keys']
        for key in keys:
            yield key
            date_from = max(date_from, to_datetime(key['updated_at']))

        has_more = len(keys) >= LIMIT
        params['date_from'] = date_from


def sync_keys():
    last_max_time = (
        SSHKey.objects
        .order_by('-modified_at')
        .values_list('modified_at', flat=True)
        .first()
    )
    all_ids = set(
        SSHKey.objects
        .values_list('id', flat=True)
    )
    to_upd = []
    to_create = []
    for key in get_keys_from_cauth(last_max_time):
        id_ = key['id']
        list_to_add = to_upd if id_ in all_ids else to_create
        list_to_add.append(SSHKey(
            id=id_,
            intranet_status=key['intranet_status'],
            created_at=to_datetime(key['created_at']),
            modified_at=to_datetime(key['updated_at']),
            description=key['description'],
            key=key['key'],
            fingerprint=key['fingerprint'],
            fingerprint_sha256=key['fingerprint_sha256'],
            staff_id=key['staff_id'],
        ))
    with atomic():
        for key in to_upd:
            key.save()
        SSHKey.objects.bulk_create(to_create)


@app.task
class SyncSshKeys(LockedTask):

    def locked_run(self):
        sync_keys()
