import logging
import re
import json

from datetime import datetime

from sshpubkeys import (
    SSHKey as SSHKey_,
    InvalidKeyException,
    TooShortKeyException,
    TooLongKeyException,
    InvalidTypeException,
    MalformedDataException,
)
from django.contrib.auth.models import AnonymousUser
from django.core.urlresolvers import reverse
from django.contrib.auth.decorators import permission_required
from django.http import (
    HttpResponseServerError,
    HttpResponseNotFound,
    HttpResponseRedirect,
    HttpResponseBadRequest,
    HttpResponse,
    JsonResponse,
)
from django.views.decorators.csrf import csrf_exempt
from django.views.decorators.http import require_POST, require_GET

from staff.lib.decorators import responding_json, responding_csv, auth_by_tvm_only
from staff.lib.forms.staff_form_grid import StaffFormGrid
from staff.person.models import Staff
from staff.person_profile.controllers import ssh_key as ssh_key_ctl
from staff.person_profile.forms.ssh_key import SSHKeyForm
from staff.keys.models import SSHKey
from staff.keys.audit import log_ssh_key_updating

logger = logging.getLogger(__name__)


@require_POST
@responding_json
@permission_required('django_intranet_stuff.can_withdraw_keys')
def revoke_ssh_keys(request, login):
    for key in SSHKey.objects.filter(staff__login=login, intranet_status=1):
        key.intranet_status = 0
        key.save(force_update=True)
        log_ssh_key_updating(
            action='removed',
            fingerprint=key.fingerprint,
            fingerprint_sha256=key.fingerprint_sha256,
            author_login=request.user.username,
            owner_login=login,
            ip=request.real_user_ip,
        )

    return HttpResponseRedirect(
        reverse('anketa-anketa', kwargs={'yandex_login': login})
    )


@require_GET
@responding_csv(filename='keys_audit')
@permission_required('django_intranet_stuff.can_see_keys_audit')
def keys_audit(request):
    bits = int(request.GET.get('bits', 2048))
    invalid = int(request.GET.get('invalid', 0))
    types = request.GET.getlist('type', ('ssh-rsa', 'ssh-dsa', 'ssh-dss'))
    dismissed = request.GET.get('dismissed')
    is_deleted = 0 if int(request.GET.get('is_deleted', 0)) else 1
    qs = (
        SSHKey.objects
        .values('id', 'key', 'staff__login', 'staff__is_dismissed', 'intranet_status')
    )
    if dismissed is not None:
        qs = qs.filter(staff__is_dismissed=dismissed)
    qs = qs.filter(intranet_status=is_deleted)
    result = []
    _re = re.compile(r'^ssh-(rsa|dsa|dss) ')
    try:
        for key in qs:
            error = ''
            login = key['staff__login']
            value = key['key']
            is_dismissed = key['staff__is_dismissed']
            employee = 'dismissed' if is_dismissed else 'employee'
            intranet_status = key['intranet_status']
            key_id = key['id']
            if _re.match(value):
                try:
                    ssh = SSHKey_(value)
                    if (bits == 0 or ssh.bits < bits) and ssh.key_type.decode() in types:
                        result.append((
                            'short',
                            'active' if intranet_status else 'deleted',
                            login,
                            key_id,
                            employee,
                            value.encode('utf-8'),
                            ssh.bits,
                        ))
                except TooShortKeyException:
                    error = 'too short'
                except TooLongKeyException:
                    error = 'too long'
                except InvalidTypeException:
                    error = 'type does not match'
                except MalformedDataException:
                    error = 'decoding and extracting the data fails'
                except InvalidKeyException:
                    error = 'invalid key at all'
                except Exception as e:
                    error = 'general error = (%s)' % e

                if error and invalid:
                    result.append((
                        'invalid',
                        'active' if intranet_status else 'deleted',
                        login,
                        key_id,
                        employee,
                        value,
                        error,
                    ))
    except Exception:
        logger.exception('Error loding keys')
        return 'Error loading keys', 500

    return result


@csrf_exempt
@require_POST
@permission_required('django_intranet_stuff.can_upload_any_ssh_key')
def upload_any_ssh_key(request, login):
    try:
        staff_id = Staff.objects.values_list('id', flat=True).get(login=login)
    except Staff.DoesNotExist:
        return HttpResponseNotFound('There are no such user')

    try:
        data = json.loads(request.body)
    except ValueError:
        return HttpResponseBadRequest('Invalid JSON')

    now = datetime.now()
    try:
        key = data['key']
    except KeyError:
        return HttpResponseBadRequest('Request without key, use {"key": "..", "description": ".."} format')

    try:
        fingerprint = ssh_key_ctl.ssh_fingerprint(key)
        fingerprint_sha256 = ssh_key_ctl.ssh_fingerprint(key, algo='sha256')
    except ssh_key_ctl.SSHKeyError:
        return HttpResponseBadRequest('Invalid SSH key')

    try:
        SSHKey(**{
            'staff_id': staff_id,
            'modified_at': now,
            'created_at': now,
            'key': key,
            'description': data.get('description', ''),
            'fingerprint': fingerprint,
            'fingerprint_sha256': fingerprint_sha256,
        }).save(force_insert=True)
        log_ssh_key_updating(
            action='added',
            fingerprint=fingerprint,
            fingerprint_sha256=fingerprint_sha256,
            author_login=request.user.username,
            owner_login=login,
            ip=request.real_user_ip,
        )
    except Exception:
        logger.exception('Cannot save SSH key: %s', key)
        return HttpResponseServerError('Ooops... Something wrong')

    return HttpResponse('Key saved')


@csrf_exempt
@require_POST
@auth_by_tvm_only(['skotty'])
def skotty_upload_ssh_keys(request):
    try:
        json_data = json.loads(request.body)
        assert isinstance(json_data, list), ValueError
    except ValueError:
        return JsonResponse(data={'success': False, 'errors': ['Invalid JSON']}, status=400)

    if isinstance(request.user, AnonymousUser):
        return JsonResponse(data={'success': False, 'errors': ['User ticket is required']}, status=400)

    person = request.user.get_profile()

    for key in json_data:
        key['person'] = person

    grid = StaffFormGrid(SSHKeyForm, data=json_data)
    if not grid.is_valid():
        return JsonResponse(
            data={'success': False, 'errors': [v['key'][0]['error_key'] for k, v in grid.errors().items()]},
            status=400,
        )
    try:
        ssh_key_ctl.add(
            key_data=grid.cleaned_data,
            person=person,
            ip=request.real_user_ip,
            author_login='skotty_tvm',
        )
    except ssh_key_ctl.SSHKeyError as e:
        return JsonResponse(
            data={'success': False, 'errors': [f'Error saving ssh keys: {e}']},
            status=400,
        )
    return JsonResponse(data={'success': True}, status=200)


@csrf_exempt
@require_POST
@auth_by_tvm_only(['skotty'])
def skotty_revoke_ssh_keys(request):
    try:
        json_data = json.loads(request.body)
        assert isinstance(json_data, dict), ValueError
    except ValueError:
        return JsonResponse(data={'success': False, 'errors': ['Invalid JSON']}, status=400)

    try:
        person = Staff.objects.get(login=json_data['login'])
    except Staff.DoesNotExist:
        return JsonResponse(data={'success': False, 'errors': ['User does not exist']}, status=400)

    try:
        ssh_key_ctl.revoke_by_fingerprints(
            key_fingerprints=json_data['fingerprints'],
            person=person,
            ip=request.real_user_ip,
            author_login='skotty_tvm',
        )
    except ssh_key_ctl.SSHKeyError as e:
        return JsonResponse(
            data={'success': False, 'errors': [f'Error revoking ssh keys: {e}']},
            status=400,
        )
    return JsonResponse(data={'success': True}, status=200)
