import logging
from collections import namedtuple

from django.utils.six import text_type
from django.views.decorators.csrf import ensure_csrf_cookie
from django.views.decorators.http import require_http_methods

from staff.lib.decorators import responding_json as responding_json_orig, consuming_json
from .controllers import AddKudoController, ViewKudoController

ViewResult = namedtuple('ViewResult', ['errors', 'data'])

logger = logging.getLogger(__name__)


def responding_json(actual_view):
    """Декоратор, оборачивающий любой ответ в json, имеющий базовые элементы."""

    @responding_json_orig
    def responding_json__(request):
        errors = []
        data = {}

        try:
            result = actual_view(request)  # type: ViewResult

        except Exception as e:
            result = ViewResult(errors=['%s' % e], data={})
            logger.exception('Kudos API exception')

        errors.extend(result.errors)
        data.update(result.data)

        response = {'success': not bool(errors)}

        if errors:
            response['errors'] = errors

        elif data:
            response['data'] = data

        return response, 400 if errors else 200

    return responding_json__


def get_bool(request, param):
    """Возвращает значение указанного GET параметра в виде булева.

    Пример:
        * '1' - True
        * 'true' - True

    :param request:
    :param str|str param: Имя параметра.
    :rtype: bool

    """
    return request.GET.get(param, '0') in {'1', 'true'}


def get_list(request, param):
    """Возвращает значение указанного GET параметра в виде списка.

    Пример:
        * '[1, 3  , 5]' - [1, 3, 5]

    :param request:
    :param str|str param: Имя параметра.
    :rtype: bool

    """
    result = [
        value.strip()
        for value in request.GET.get(param, '').strip('[]').split(',')
        if value.strip()]
    return result


def users_handling(actual_view):
    """Декоратор для представлений, выдающих информацию о пользователях,
    логины которых переданы на вход.

    """
    @responding_json
    def users_request_handler_(request):

        users = get_list(request, 'users')

        if users:
            result = actual_view(request, users)  # type: ViewResult

        else:
            result = ViewResult(
                errors=["Field 'users' containing users login list is mandatory"],
                data={},
            )

        return result

    return users_request_handler_


def contribute_log_entries(user_stats, key, entries):
    """Добавляет в элементы контейнера со статистикой
    по пользователю записи из журнала благодарностей по указанному ключу.

    :param dict user_stats:
    :param str|str key:
    :param dict entries:

    """
    for user_login, log_entries in entries.items():
        log = []

        for log_entry in log_entries:
            log.append({
                'createdAt': log_entry['created_at'],
                'issuer': log_entry['issuer__login'],
                'recipient': log_entry['recipient__login'],
                'power': log_entry['power'],
                'message': log_entry['message'],
            })

        user_stats[user_login][key] = log


@ensure_csrf_cookie
@require_http_methods(['GET'])
@users_handling
def view(request, users):
    """Возвращает данные о благодарностях для указанных пользователей.

    :param request:
    :param list[str|str] users:
    :rtype: tuple[list, dict]

    """
    show_log_taken = get_bool(request, 'logTaken')
    show_log_given = get_bool(request, 'logGiven')
    show_available = get_bool(request, 'available')

    user_stats = {}
    data = {'stats': user_stats}
    errors = []

    controller = ViewKudoController

    if show_available:
        data['available'] = controller.get_available(
            recipients=users, issuer=request.user.get_profile())

    stats = controller.get_stats(users=users)

    for stat in stats:
        user_stats[stat['person__login']] = {
            'score': text_type(stat['score']),
        }

    # Нормализуем, добавляя данные по умолчанию
    # для записей, отсутствоваших в БД.
    for user_login in users:

        record = user_stats.setdefault(user_login, {
            'score': '0',
        })

        if show_log_given:
            record.setdefault('logGiven', [])

        if show_log_taken:
            record.setdefault('logTaken', [])

    # Добавляем данные журналов, если запрошены.
    if show_log_taken or show_log_given:
        given, taken = controller.get_log(
            users=users,
            given=show_log_given,
            taken=show_log_taken,
        )

        contribute_log_entries(user_stats, key='logGiven', entries=given)
        contribute_log_entries(user_stats, key='logTaken', entries=taken)

    return ViewResult(errors, data)


@ensure_csrf_cookie
@require_http_methods(['POST'])
@responding_json
@consuming_json
def add(request, json_data):
    """Отвечает за начисление одобрения от текущего пользователя указанным.

    Ожидается json вида:
        {
            "users": ["userlogin1", "userlogin2"],
            "message": "You are so cool!",
            "notify": true
        }

    .. note:: Благодарность не зачисляется:
        * неизвестным логинам
        * уволенным и от уволенных
        * от дающего себе же
        * скорые повторные выдачи одному и тому же получателю
        * от виртуальных пользователей

    :param request:
    :param dict json_data:
    :rtype: dict

    """
    errors = []

    recipients = json_data.get('users')
    message = json_data.get('message', '')
    notify = bool(json_data.get('notify', False))

    if recipients:

        AddKudoController.add_kudo(
            issuer=request.user.get_profile(),
            recipients=recipients,
            message=message,
            notify=notify,
        )

    else:
        errors.append("Field 'users' containing users login list is mandatory")

    return ViewResult(errors=errors, data={})
