from typing import Any, Dict, Iterable

from django.conf import settings
from django.db.models import Q, F
from django.http import Http404

from staff.lib.models.departments_chain import get_departments_tree
from staff.lib.utils.dict import invert_multivaluedict
from staff.lib.utils.qs_values import localize

from staff.departments.models import Department
from staff.lenta.forms import LentaParamsForm
from staff.lenta.models import LentaLog, ACTION_CHOICES, LAST_RECORDS_LIMIT
from staff.person.models import MemorialProfile


type_to_action = {
    'hired': [ACTION_CHOICES.HIRED, ACTION_CHOICES.RETURNED],
    'dismissed': [ACTION_CHOICES.DISMISSED],
    'changed': [
        ACTION_CHOICES.TRANSFERRED,
        ACTION_CHOICES.CHANGED_POSITION,
        ACTION_CHOICES.TRANSFERRED_AND_CHANGED_POSITION,
    ],
}
action_to_type = invert_multivaluedict(type_to_action)


ACTION_RELEVANT_FIELDS = {
    ACTION_CHOICES.HIRED: [
        'position_new',
        'department_new',
        'office_new',
        'organization_new',
    ],
    ACTION_CHOICES.DISMISSED: [
        'position_old',
        'department_old',
        'office_old',
        'organization_old',
    ],
    ACTION_CHOICES.CHANGED_POSITION: [
        'position_old',
        'position_new',
    ],
    ACTION_CHOICES.TRANSFERRED: [
        'department_old',
        'department_new',
    ],
    ACTION_CHOICES.TRANSFERRED_AND_CHANGED_POSITION: [
        'position_old',
        'department_old',
        'position_new',
        'department_new',
    ],
    ACTION_CHOICES.RETURNED: [
        'position_old',
        'position_new',
        'department_old',
        'department_new',
        'office_old',
        'office_new',
        'organization_old',
        'organization_new',
    ],
    ACTION_CHOICES.MOVED: [
        'office_old',
        'office_new',
    ],
    ACTION_CHOICES.CHANGED_ORGANIZATION: [
        'organization_old',
        'organization_new',
    ]
}

PRIVATE_ACTIONS = (
    ACTION_CHOICES.CHANGED_HEADCOUNT,
)


def get_records(params_form: LentaParamsForm):

    action_type = params_form.cleaned_data['action_type']
    continuation_token = params_form.cleaned_data['continuation_token']
    department_id = params_form.cleaned_data['department_id']
    ignore_descendants = params_form.cleaned_data['ignore_descendants']

    yandex_tree_id = Department.objects.get(id=settings.YANDEX_DEPARTMENT_ID).tree_id
    action_filter = type_to_action.get(action_type, action_to_type.keys())
    changed_position = ACTION_CHOICES.CHANGED_POSITION
    memorial_staff_ids = MemorialProfile.objects.all().values_list('person_id', flat=True)

    qs = (
        LentaLog.objects
        .filter(
            ~Q(action=changed_position) | (Q(action=changed_position) & ~Q(position_old__iexact=F('position_new'))),
            action__in=action_filter,
            department_new__isnull=False,
            department_new__tree_id=yandex_tree_id,
        )
    )

    if department_id is not None:
        try:
            if ignore_descendants:
                Department.objects.get(id=department_id)
                departments = {department_id}
            else:
                departments = (
                    Department.objects
                    .get(id=department_id)
                    .get_descendants(include_self=True)
                    .values_list('id', flat=True)
                )
            qs = qs.filter(Q(department_new_id__in=departments) | Q(department_old_id__in=departments))
        except Department.DoesNotExist:
            raise Http404

    if continuation_token is not None:
        try:
            prev_item = LentaLog.objects.get(pk=continuation_token)
            qs = qs.filter(
                Q(created_at__lt=prev_item.created_at)
                | (Q(created_at=prev_item.created_at) & Q(pk__lt=prev_item.pk))
            )
        except LentaLog.DoesNotExist:
            raise Http404

    records = (
        qs
        .exclude(staff__in=memorial_staff_ids)
        .select_related('staff')
        [:LAST_RECORDS_LIMIT]
    )
    return records


def get_records_as_dict(params_form: LentaParamsForm):

    records = get_records(params_form)

    departments_ids = []
    for record in records:
        if record.department_old_id is not None:
            departments_ids.append(record.department_old_id)
        departments_ids.append(record.department_new_id)

    chains = get_departments_tree(
        departments=departments_ids,
        fields=['name', 'name_en', 'url', 'id'],
    )

    result = []

    for record in records:
        record_dict = {
            'id': record.id,
            'date': record.created_at.date().isoformat(),
            'login': record.staff.login,
            'gender': record.staff.gender,
            'first_name': record.staff.first_name,
            'last_name': record.staff.last_name,
            'first_name_en': record.staff.first_name_en,
            'last_name_en': record.staff.last_name_en,
            'action': ACTION_CHOICES.get_name(record.action),
            'type': action_to_type[record.action][0],
        }

        AC = ACTION_CHOICES

        if record.action not in (AC.CHANGED_POSITION, AC.DISMISSED):
            record_dict['department_new'] = [
                localize(dep) for dep in chains.get(record.department_new_id, [])
            ]

        if record.action not in (AC.CHANGED_POSITION, AC.HIRED, AC.RETURNED):
            record_dict['department_old'] = [
                localize(dep) for dep in chains.get(record.department_old_id, [])
            ]

        if record.action not in (AC.TRANSFERRED, AC.HIRED, AC.RETURNED):
            record_dict['position_old'] = record.position_old

        if record.action not in (AC.TRANSFERRED, AC.DISMISSED):
            record_dict['position_new'] = record.position_new

        result.append(localize(record_dict))

    return result


def get_person_records(login):
    records = LentaLog.objects.filter(
        staff__login=login,
    ).exclude(
        action__in=PRIVATE_ACTIONS,
    ).select_related(
        'department_old',
        'department_new',
        'office_old',
        'office_new',
        'organization_old',
        'organization_new',
    ).order_by('-created_at')

    person_records = []

    for record in records:
        record_data = {
            'date': record.created_at.date().isoformat(),
            'action': ACTION_CHOICES.get_name(record.action),
        }
        record_data.update(fetch_action_fields(record))
        person_records.append(record_data)

    return person_records


def fetch_action_fields(record: LentaLog) -> Dict[str, Any]:
    fields = ACTION_RELEVANT_FIELDS[record.action]

    data = {}
    for field in fields:
        value = getattr(record, field)

        if field.startswith('department'):
            serialized = _get_generic_serialized_value(value, ['url'])
        elif field.startswith('office'):
            serialized = _get_generic_serialized_value(value, ['code', 'filter_id'])
        elif field.startswith('organization'):
            serialized = _get_generic_serialized_value(value, ['filter_id'])
        else:
            serialized = value

        data[field] = serialized

    return data


def _get_generic_serialized_value(value: Any, additional_fields: Iterable[str]) -> Dict[str, Any]:
    if value is None:
        return {'id': None, 'name': {'ru': None, 'en': None}}

    serialized = {
        'id': value.id,
        'name': {
            'ru': value.name,
            'en': value.name_en,
        },
    }

    for field in additional_fields:
        serialized[field] = getattr(value, field)

    return serialized
