from datetime import datetime

from django.db import models
from django.dispatch import receiver

from staff.person.models import Staff, StaffExtraFields
from staff.person.memorial import Memorial

from staff.departments.utils import get_department_path
from staff.lib.utils.date import get_timedelta_setting

from .models import LentaLog, ACTION_CHOICES


def store_old_data(sender, instance, **kwargs):
    """
    Метод получает сигнал перед сохранением модели и
    сохраняет её экземпляр до изменения в атрибут _old_instance.
    """
    if instance.pk is None:
        instance._old_instance = None
    else:
        try:
            instance._old_instance = sender.objects.get(id=instance.id)
        except sender.DoesNotExist:
            instance._old_instance = None


models.signals.pre_save.connect(store_old_data, sender=Staff)
models.signals.pre_save.connect(store_old_data, sender=StaffExtraFields)


@receiver(models.signals.post_save, sender=StaffExtraFields)
def person_extra_fields_log_changes(sender, instance, **kwargs):

    old = instance._old_instance
    new = instance

    old_oebs_headcount = False if old is None else old.oebs_headcount

    if old_oebs_headcount == new.oebs_headcount:
        return

    person = new.staff

    LentaLog.objects.create(
        staff=person,
        action=ACTION_CHOICES.CHANGED_HEADCOUNT,
        position_new=person.position,
        department_new_id=person.department_id,
        department_new_path=get_department_path(person.department),
        office_new_id=person.office_id,
        organization_new_id=person.organization_id,
        position_old=person.position,
        department_old_id=person.department_id,
        office_old_id=person.office_id,
        organization_old_id=person.organization_id,
        headcount_old=old_oebs_headcount,
        headcount_new=new.oebs_headcount,
    )


@receiver(models.signals.post_save, sender=Staff)
def person_log_changes(sender, instance, **kwargs):
    """
    Метод получает сигнал после сохранения моделей Staff и
    делает запись в логе, если изменения подходят для вывода в ленту.
    Экземпляр до изменений сохранен в атрибуте new._old_instance
    """
    new = instance
    old = new._old_instance

    if old is None:
        action = ACTION_CHOICES.HIRED
        position_old = ''
        department_old_id = None
        office_old_id = None
        organization_old_id = None
    else:
        position_old = old.position
        department_old_id = old.department_id
        office_old_id = old.office_id
        organization_old_id = old.organization_id

        if new.is_dismissed and not old.is_dismissed:
            action = ACTION_CHOICES.DISMISSED
        elif not new.is_dismissed and old.is_dismissed:
            action = ACTION_CHOICES.RETURNED
        elif position_old != new.position and department_old_id != new.department_id:
            action = ACTION_CHOICES.TRANSFERRED_AND_CHANGED_POSITION
        elif position_old != new.position:
            action = ACTION_CHOICES.CHANGED_POSITION
        elif department_old_id != new.department_id:
            action = ACTION_CHOICES.TRANSFERRED
        elif new.office_id != office_old_id:
            action = ACTION_CHOICES.MOVED
        elif new.organization_id != organization_old_id:
            action = ACTION_CHOICES.CHANGED_ORGANIZATION
        else:
            return

    if action == ACTION_CHOICES.DISMISSED and Memorial(new):
        return

    if action == ACTION_CHOICES.RETURNED:
        end = datetime.now()
        start = end - get_timedelta_setting('LENTA_UNFIRE_INTERVAL')
        try:
            LentaLog.objects.get(
                staff_id=new.id,
                action=ACTION_CHOICES.DISMISSED,
                created_at__range=(start, end)
            ).delete()
            return  # Annihilate recent dismission with return.
        except LentaLog.DoesNotExist:
            pass

    data = dict(
        staff=new,
        action=action,
        position_new=new.position,
        department_new_id=new.department_id,
        department_new_path=get_department_path(new.department),
        office_new_id=new.office_id,
        organization_new_id=new.organization_id,
        position_old=position_old,
        department_old_id=department_old_id,
        office_old_id=office_old_id,
        organization_old_id=organization_old_id,
    )

    LentaLog.objects.create(**data)
