from staff.reports.objects.base import Report

from ..models import LentaLog, ACTION_CHOICES

ACTIONS = {
    'h': 'Найм',
    'd': 'Увольнение',
    'c': 'Смена должности',
    't': 'Переезд',
    'x': 'Переезд и смена должности',
    'r': 'Возвращение',
    'm': 'Перемещение',
    'o': 'Смена организации',
    'e': 'Смена Headcount',
}


def str_date(date):
    return date.strftime('%d.%m.%Y') if date else ''


class DateFilteredLentaReport(Report):
    """Абстрактный класс отчёта из ленты, отфильтрованного по датам"""
    sheet_name = 'lenta'
    target_actions = [action.value for action in ACTION_CHOICES._options]

    lenta_fields = None

    def __init__(self, date_from, date_to, *args, **kwargs):
        self.date_from = date_from
        self.date_to = date_to
        super(DateFilteredLentaReport, self).__init__(*args, **kwargs)

    def process_log_entry(self, log_entry):
        log_entry['login'] = log_entry.pop('staff__login')
        log_entry['person'] = '{}, {}'.format(
            log_entry.pop('staff__last_name'),
            log_entry.pop('staff__first_name'),
        )
        return log_entry

    def join_data(self):

        lenta_log = (
            LentaLog.objects.values(*self.lenta_fields)
            .filter(
                created_at__lte=self.date_to.isoformat(),
                created_at__gte=self.date_from.isoformat(),
                action__in=self.target_actions,
            )
            .order_by('created_at')
        )

        def normalize_linked_instances(record):
            for field_name in record:
                if field_name.endswith('__name'):
                    record[field_name[:-6]] = record.pop(field_name)
            return record

        for log_entry in lenta_log:
            log_entry = self.process_log_entry(log_entry)
            yield normalize_linked_instances(log_entry)


class HeadcountReport(DateFilteredLentaReport):
    target_actions = [ACTION_CHOICES.CHANGED_HEADCOUNT, ]

    def get_head_rows(self):
        yield [
            'Дата',
            'Логин',
            'ФИО',
            'Старое знач. Headcount',
            'Новое знач. Headcount',
        ]

    lenta_fields = [
        'staff__login', 'staff__first_name', 'staff__last_name',
        'created_at',
        'headcount_old', 'headcount_new'
    ]

    def process_log_entry(self, log_entry):
        log_entry = super(HeadcountReport, self).process_log_entry(log_entry)
        log_entry['headcount_old'] = 'да' if log_entry['headcount_old'] else 'нет'
        log_entry['headcount_new'] = 'да' if log_entry['headcount_new'] else 'нет'
        return log_entry

    def get_rows(self):
        for log_entry in self.join_data():
            yield [
                str_date(log_entry['created_at']),
                log_entry['login'],
                log_entry['person'],
                log_entry['headcount_old'],
                log_entry['headcount_new'],
            ]


class LentaReport(DateFilteredLentaReport):
    target_actions = [
        action.value for action in ACTION_CHOICES._options
        if action.value != ACTION_CHOICES.CHANGED_HEADCOUNT
    ]

    def get_head_rows(self):
        yield [
            'Дата',
            'Логин',
            'ФИО',
            'Событие (тип перемещения)',
            'Старая Должность',
            'Новая должность',
            'Старое подразделение',
            'Новое подразделение',
            'Старый офис',
            'Новый офис',
            'Старая организация',
            'Новая организация',
        ]

    lenta_fields = [
        'action',
        'staff__login', 'staff__first_name', 'staff__last_name',
        'created_at',
        'position_old', 'position_new',
        'department_old__name', 'department_new__name',
        'office_old__name', 'office_new__name',
        'organization_old__name', 'organization_new__name',
    ]

    def process_log_entry(self, log_entry):
        log_entry = super(LentaReport, self).process_log_entry(log_entry)
        log_entry['action'] = ACTIONS[log_entry['action']]
        return log_entry

    def get_rows(self):
        for log_entry in self.join_data():
            yield [
                str_date(log_entry['created_at']),
                log_entry['login'],
                log_entry['person'],
                log_entry['action'],
                log_entry['position_old'],
                log_entry['position_new'],
                log_entry['department_old'],
                log_entry['department_new'],
                log_entry['office_old'],
                log_entry['office_new'],
                log_entry['organization_old'],
                log_entry['organization_new'],
            ]
