from collections import OrderedDict
from datetime import date
from itertools import groupby
from operator import itemgetter

from django import http
from django.conf import settings

from staff.lib.models.base import get_i_field
from staff.lib.decorators import front_json
from staff.lib.utils.qs_values import extract_related, localize
from staff.map.models import City

from staff.lenta.feeds import get_records_as_dict, get_person_records
from staff.lenta.forms import ChartForm, COUNTRIES_BY, FreeChartForm, GROUP_BY, LentaParamsForm
from staff.lenta.models import CountAndGrowth
from staff.lenta.objects import FlexChart


def lenta_data(request):
    """Лента кадровых изменений"""
    params_form = LentaParamsForm(data=request.GET)
    if not params_form.is_valid():
        return http.HttpResponse(params_form.errors)

    log_records = get_records_as_dict(params_form)

    result = OrderedDict()
    for record_date, day_records in groupby(log_records, lambda r: r['date']):
        result[record_date] = list(day_records)

    return http.JsonResponse(data=result, safe=False)


def lenta_personal(request, login):
    return http.JsonResponse(data=get_person_records(login=login), safe=False)


# @cache_page(60 * 15)
@front_json
def flex_chart(request):
    form = ChartForm(request.GET)
    if not form.is_valid():
        return {
            'errors': [{
                'code': 'validation_error',
                'desc': 'Some parameters are invalid.'
            }],
            'content': form.errors,
        }

    value_chart_map = {
        'total': lambda p, v: v['delta'] != 0 or p == 0,
        'plus': lambda p, v: v['plus'] != 0,
        'minus': lambda p, v: v['minus'] != 0,
    }

    chart = FlexChart(**form.cleaned_data)
    result = {}
    for period, city_or_country, values in chart:
        for field, condition in value_chart_map.items():
            collection = result.setdefault(field, {})
            geo_data = collection.setdefault(city_or_country, {})
            if condition(period, values):
                geo_data[period] = values[field]

    result.update({
        'value_keys': chart.value_fields,
        'cities' if chart.group_by == GROUP_BY.city else 'countries':
            list(chart.get_used_geos()),
        'delta': chart.grouped_delta,
        'params': dict(request.GET),
    })
    return result


def smart_isodate(date_or_string):
    if isinstance(date_or_string, str):
        string = date_or_string
    else:
        string = date_or_string.isoformat()
    return string


class ChartMetaViewParamsMapper:
    form_class = FreeChartForm

    field_defaults = {
        'min_date': lambda self: settings.CHART_START_DATE,
        'max_date': lambda self: date.today(),
        'countries_by': lambda self: None,
        'departments': lambda self: None,
        'countries': lambda self: None,
        'with_homies': lambda self: False,
        'only_headcount': lambda self: False,
    }

    def __init__(self, request_params):
        self.form = self.form_class(request_params)
        self.is_valid = None

    def as_dict(self):
        return {
            field: getattr(self, field, default(self))
            for field, default in self.field_defaults.items()
        }

    @property
    def min_date(self):
        self._validate()
        return smart_isodate(self.form.cleaned_data['start'])

    @property
    def max_date(self):
        self._validate()
        return smart_isodate(self.form.cleaned_data['end'])

    @property
    def countries_by(self):
        self._validate()
        return [
            {'name': k, 'value': v, 'selected': True}
            if k in self._selected_countries_by
            else {'name': k, 'value': v}
            for k, v in COUNTRIES_BY
        ]

    @property
    def departments(self):
        self._validate()
        return self._get_departments()

    @property
    def countries(self):
        self._validate()
        return self._get_countries()

    @property
    def with_homies(self):
        return self.form.cleaned_data['with_homies']

    @property
    def only_headcount(self):
        return self.form.cleaned_data['only_headcount']

    # private

    def _validate(self):
        self.is_valid = self.form.is_valid()

    @property
    def _selected_countries_by(self):
        if not self.is_valid:
            return []

        return [self.form.cleaned_data['countries_by']]

    @property
    def _selected_departments_ids(self):
        if not self.is_valid:
            return set()

        dep0 = self.form.cleaned_data['department_0']
        dep1 = self.form.cleaned_data['department_1']
        ids = (getattr(dep, 'id', None) for dep in (dep0, dep1))
        return {id_ for id_ in ids if id_ is not None}

    @property
    def _selected_cities_ids(self):
        if not self.is_valid:
            return set()

        return {city.id for city in self.form.cleaned_data['cities']}

    def _get_departments(self):
        def wrap_department(dep):
            dep['is_active'] = dep.pop('intranet_status') == 1
            if dep['id'] in self._selected_departments_ids:
                dep['selected'] = True
            return dep

        deps_with_data = (
            CountAndGrowth.objects
            .values(
                'department_0__id',
                'department_0__name',
                'department_0__name_en',
                'department_0__intranet_status',
                'department_1__id',
                'department_1__name',
                'department_1__name_en',
                'department_1__intranet_status',
            )
            .order_by(
                'department_0',
                'department_1__lft',
            )
            .distinct()
        )

        departments = []
        cur_dep_0_id = None

        for dep in deps_with_data:
            dep = localize(dep)
            dep_0_id = dep['department_0__id']

            if dep_0_id != cur_dep_0_id:
                cur_dep_0_id = dep_0_id
                dep_0 = wrap_department(extract_related(dep, 'department_0'))
                dep_0['children'] = []
                departments.append(dep_0)

            if dep['department_1__id']:
                dep_1 = wrap_department(extract_related(dep, 'department_1'))
                departments[-1]['children'].append(dep_1)

        return departments

    @property
    def _countries_by(self):
        if self.is_valid:
            return self.form.cleaned_data['countries_by'] or COUNTRIES_BY.office
        else:
            return COUNTRIES_BY.office

    def _get_cities_with_data(self):
        field_name = '%s__city' % self._countries_by
        return (
            CountAndGrowth.objects
            .exclude(**{field_name: None})
            .values_list(field_name, flat=True)
            .iterator()
        )

    def _get_countries(self):
        cities_with_data = self._get_cities_with_data()
        city_fields = [
            'id', 'native_lang', 'name', 'name_en', 'intranet_status', 'color'
        ]
        fields = ['country__' + f for f in city_fields] + city_fields
        cities_qs = (
            City.objects
            .filter(id__in=cities_with_data)
            .values(*fields)
            .order_by('country__position', 'country', 'position')
        )

        def wrap(dict_, prefix=''):
            return {
                'id': dict_[prefix + 'id'],
                'name': get_i_field(dict_, 'name', prefix),
                'is_active': dict_[prefix + 'intranet_status'] == 1,
                'color': dict_[prefix + 'color'],
            }

        def wrap_city(city_dict, prefix=''):
            dict_ = wrap(city_dict, prefix)
            if dict_['id'] in self._selected_cities_ids:
                dict_['selected'] = True
            return dict_

        countries = []
        for country_id, cities in groupby(cities_qs, itemgetter('country__id')):
            city = next(cities)
            country = wrap(city, prefix='country__')
            country['cities'] = []
            country['cities'].append(wrap_city(city))
            for city in cities:
                country['cities'].append(wrap_city(city))
            countries.append(country)
        return countries


@front_json
def chart_meta(request):
    return ChartMetaViewParamsMapper(request.GET).as_dict()
