from typing import AnyStr, Union, Dict
from django.contrib.auth.backends import ModelBackend
from django.contrib.auth.models import Permission, Group as DjGroup
from django.db.models import Q
from django.core.cache import cache

from staff.departments.models import DepartmentRole
from staff.groups.models import Group
from staff.users.models import User


class IntranetBackend(ModelBackend):
    """
    Бэкенд добавляет права для групп, в которые входят джанго-пользователь к
    множеству уже имеющихся у него прав. Для использования нужно вписать в
    настройки:
    AUTHENTICATION_BACKENDS = ('django_intranet_stuff.auth_backends.IntranetBackend',)
    """

    def get_all_permissions(self, user_obj, obj=None):
        if user_obj.is_anonymous():
            return set()

        if user_obj.get_profile().is_dismissed:
            return set()

        if not hasattr(user_obj, '_perm_cache'):
            key = f'GLOBAL_ALL_PERMS_{user_obj.username}'
            user_obj._perm_cache = cache.get(key)
            if user_obj._perm_cache is None:
                qs = self._permissions_query(user_obj).values_list('content_type__app_label', 'codename')
                user_obj._perm_cache = set("%s.%s" % perm for perm in qs)
                cache.set(key, user_obj._perm_cache, 60 * 10)
        return user_obj._perm_cache

    def get_group_permissions(self, user_obj, obj=None):
        return self.get_all_permissions(user_obj, obj)

    def _permissions_query(self, user_obj):
        result = Permission.objects.order_by()  # Disables default ordering that forces unnecessary join

        if user_obj.is_superuser:
            return result

        staff_group_permissions_q = Q(id__in=Group.objects.filter(members=user_obj.get_profile()).values('permissions'))
        django_group_permissions_q = Q(id__in=DjGroup.objects.filter(user=user_obj).values('permissions'))
        user_permissions_q = Q(id__in=User.objects.filter(id=user_obj.id).values('user_permissions'))
        roles_permissions_q = Q(id__in=(
            DepartmentRole.objects
            .filter(holders=user_obj.get_profile())
            .values('permissions')
        ))

        filter_q = (
            staff_group_permissions_q |
            django_group_permissions_q |
            user_permissions_q |
            roles_permissions_q
        )

        return result.filter(filter_q)

    def has_perm(self, user_obj, perm, obj=None):
        if user_obj.is_superuser:
            return True

        if obj is not None:
            return self._check_permission_for_department(user_obj, perm, obj)

        return super(IntranetBackend, self).has_perm(user_obj, perm, obj)

    def _check_permission_for_department(self, user_obj, perm, department):
        # type: (User, AnyStr, Union[Department, Dict]) -> bool
        from staff.departments.models import Department

        department_id = department.id if isinstance(department, Department) else department['id']

        result = (
            Department.objects
            .filter(id=department_id)
            .filter(user_obj.get_profile().departments_by_perm_query(perm, True))
            .exists()
        )
        return result
