import logging
from datetime import datetime

from django.conf import settings

from staff.users.models import User
from staff.person.models import Staff
from staff.lib.auth.exceptions import (
    PersonNotFound,
    PersonIsDismissed,
    UserNotFoundError,
)

from staff.lib.db import atomic


logger = logging.getLogger('auth.utils')


def get_user_by_uid(uid):
    try:
        person = (
            Staff.objects
            .select_related('user')
            .get(uid=uid)
        )
    except Staff.DoesNotExist:
        logger.warning(
            ('Person with uid "%s" exists,'
             ' but is not found in the system'),
            uid
        )
        raise PersonNotFound

    if getattr(person, 'is_dismissed', False):
        logger.warning(
            'Person with login "%s" dismissed, but tried to log in',
            person.login,
        )
        raise PersonIsDismissed

    user = person.user

    if user is None:
        logger.error(
            ('Could not get user.'
             ' Create django user for staff with id "%s".'),
            person.pk
        )
        raise UserNotFoundError

    # Наверное так делать не хорошо, но это же -1 запрос.
    # Не писать суда lambda, так как lambda не пиклится.
    user._profile_cache = person
    return user


@atomic
def get_or_create_test_user():
    """
    Вернуть профиль по логину, уже существующий или новый.
    Пишет в базу, поэтому не работает если джангой уже выбрана реплика.
    """
    login = getattr(settings, 'AUTH_TEST_USER', 'tester')
    try:
        person = Staff.objects.get(login=login)
    except Staff.DoesNotExist:
        lang_ui = getattr(settings, 'LANGUAGE_CODE', 'en-us')[:2]
        user_model = User

        try:
            user = user_model.objects.get(username=login)
        except user_model.DoesNotExist:
            user = user_model.objects.create_user(
                login,
                '{login}@yandex-team.ru'.format(login=login)
            )
        now = datetime.now()
        from staff.lib.testing import StaffFactory
        person = StaffFactory(
            created_at=now,
            first_name=login,
            first_name_en=login,
            lang_ui=lang_ui,
            login=login,
            modified_at=now,
            native_lang='en',
            tz='Europe/Moscow',
            user=user,
        )

        user._profile_cache = person

    return person.user
