from typing import Any, Dict, List

from django.db import connection, models


def execute_bulk_update(
    target_table: models.Model,
    key_columns: List[str],
    updatable_fields: List[str],
    values: List[List[Any]]
) -> None:
    table_info = target_table._meta
    field_type_mapping = {field.name: field.db_type(connection) for field in table_info.fields}
    query = _make_update_query(table_info.db_table, field_type_mapping, key_columns, updatable_fields, len(values))
    cursor = connection.cursor()
    params = [value for row in zip(*values) for value in row]
    return cursor.execute(query, params)


def _make_update_query(
    table_name: str,
    field_type_mapping: Dict[str, str],
    key_columns: List[str],
    updatable_fields: List[str],
    rows_count: int,
) -> str:
    assert rows_count > 0
    column_types = set(field_type_mapping.values())
    args_templates = {db_type: ','.join((f'CAST(%s AS {db_type})', ) * rows_count) for db_type in column_types}

    updated_data_table_name = 'updated_data_table'
    all_rows_template = ','.join(
        f'unnest(array[{args_templates[field_type_mapping[field]]}]) AS {field}'
        for field in updatable_fields + key_columns
    )
    fields_string = ','.join([f'{field} = {updated_data_table_name}.{field}' for field in updatable_fields])
    key_matchers = (f'{table_name}.{key_column} = {updated_data_table_name}.{key_column}' for key_column in key_columns)
    where_string = ' AND '.join(key_matchers)

    query = f'''UPDATE {table_name}
        SET {fields_string}
        FROM (SELECT {all_rows_template}) AS {updated_data_table_name}
        WHERE {where_string};
    '''

    return query
