from django import forms
from django.utils.translation import ugettext as _

from staff.lib.forms.fields import StaffField
from staff.lib.forms.fields import UserFoundMultiple
from staff.lib.forms.fields import UserNotFound
from staff.person.models import Staff


class FloorModelChoiceField(forms.ModelChoiceField):
    """
    ModelChoice для кастомизации вывода названий этажей в выпадающих списках
    """
    def label_from_instance(self, obj):
        return '%s | %s' % (obj.office, obj)


class DepartmentModelChoiceField(forms.ModelChoiceField):
    """
    ModelChoice для иерархического вывода названий подрийазделен в выпадающих
    списках
    """
    def label_from_instance(self, obj):
        return '%s %s' % ('-- '*obj.level, obj)


class DashedNullBooleanSelect(forms.NullBooleanSelect):
    """
    NullBooleanSelect с изменяемым списком значений
    """
    def __init__(self, attrs=None, choices=()):
        if not choices:
            choices = (('1', '--------'),
                       ('2', _('Yes')),
                       ('3', _('No')))
        super(forms.NullBooleanSelect, self).__init__(attrs, choices)


class DashedNullBooleanField(forms.NullBooleanField):
    widget = DashedNullBooleanSelect


class StaffAutocompleteField(StaffField):

    def __init__(self, include_dismissed=False, **kw):
        self.include_dismissed = include_dismissed
        super(StaffAutocompleteField, self).__init__(**kw)

    def clean_from_text(self, input_text):
        try:
            staff_obj = self.search_staff_exactly(
                input_text,
                include_dismissed=self.include_dismissed
            )
        except (UserNotFound, UserFoundMultiple):
            raise forms.ValidationError(self.error_messages['invalid_choice'])
        else:
            return staff_obj

    def search_staff_exactly(self, query, include_dismissed=False):
        """
        Search exact fit user by query (login, name or surname)
        Used in interfaces as defense of user's autocomlete from incorrect input
        """
        if not include_dismissed:
            qs = Staff.objects.filter(is_dismissed=False)
        else:
            qs = Staff.objects.all()
        try:
            if len(query.split(' ')) == 2:
                t = query.split(' ')
                try:
                    r = qs.get(first_name=t[0], last_name=t[1])
                except Staff.DoesNotExist:
                    try:
                        r = qs.get(first_name=t[1], last_name=t[0])
                    except Staff.DoesNotExist:
                        raise UserNotFound('User "%s" has not been found' % query)
            else:
                try:
                    r = qs.get(login=query)
                except Staff.DoesNotExist:
                    try:
                        r = qs.get(last_name=query)
                    except Staff.DoesNotExist:
                        try:
                            r = qs.get(first_name=query)
                        except Staff.DoesNotExist:
                            raise UserNotFound('User "%s" has not been found' % query)
        except Staff.MultipleObjectsReturned:
            raise UserFoundMultiple('There are several users named "%s"' % query)
        return r
