import copy

from django import forms
from django.core.exceptions import ValidationError
from django.utils.safestring import mark_safe

from staff.lib.forms.widgets import AutocompleteWidget, StaffUserAutocompleteWidget

from staff.person.models import Staff, UserNotFound, UserFoundMultiple
from staff.groups.models import Group
from staff.departments.models import Department
from staff.map.models import Table

from django.core import validators


MULTIC_ENTITIES = {
    'staff': {'model': Staff, 'name': str},
    'department': {'model': Department, 'name': str},
    'table': {'models': Table, 'name': lambda x: str(x.num)},
}


MODEL_NAME = {
    Staff: 'staff',
    Department: 'department',
    Table: 'table',
}


class MulticWidget(forms.MultiWidget):
    def __init__(self, attrs=None):
        widgets = (
            forms.TextInput(attrs=attrs),
            forms.TextInput(attrs=attrs),
            forms.TextInput(attrs=attrs),
        )
        super(MulticWidget, self).__init__(widgets, attrs)

    def decompress(self, value):
        if value:
            model_name = MODEL_NAME[type(value)]
            return [model_name,
                    value.id,
                    MULTIC_ENTITIES[model_name]['name'](value)]

        return [None, None, None]

    def render(self, name, value, attrs=None):
        if not isinstance(value, list):
            value = self.decompress(value)
        output = []
        final_attrs = self.build_attrs(attrs)
        id_ = final_attrs.get('id', None)
        names = ('__type', '__value', '')
        for i, widget in enumerate(self.widgets):
            try:
                widget_value = value[i]
            except IndexError:
                widget_value = None
            if id_:
                final_attrs = dict(final_attrs, id=name + names[i])
            output.append(widget.render(name + names[i], widget_value, final_attrs))
        return mark_safe(self.format_output(output))

    def value_from_datadict(self, data, files, name):
        names = ('__type', '__value', '')
        return [
            widget.value_from_datadict(data, files, name + names[i])
            for i, widget in enumerate(self.widgets)
        ]


class MulticField(forms.MultiValueField):
    widget = MulticWidget

    def __init__(self, *args, **kwargs):
        fields = (
            # forms.ChoiceField(choices=zip(MULTIC_ENTITIES, MULTIC_ENTITIES)),
            forms.CharField(),  # type
            forms.CharField(),  # value (т.е. id)
            forms.CharField(),  # текстовое представление
        )
        super(MulticField, self).__init__(fields, *args, **kwargs)

    def compress(self, data_list):
        if data_list:
            if data_list[0] in validators.EMPTY_VALUES:
                raise ValidationError(self.error_messages['invalid_type'])
            if data_list[1] in validators.EMPTY_VALUES:
                return None
            data_type = MULTIC_ENTITIES[data_list[0]]['model']
            return data_type.objects.get(pk=data_list[1])
        return None


class MulticModelChoiceField(forms.ModelChoiceField):
    def __init__(self, *args, **kwargs):
        super(MulticModelChoiceField, self).__init__(*args, **kwargs)
        model = self.queryset.model
        model_name = MODEL_NAME[model]
        self.presentation_function = MULTIC_ENTITIES[model_name]['name']
        self.multic_type = model_name


class AUTOCOMPLETE_GROUP_TYPE:
    ALL = 'all'
    DEPARTMENT = 'departments'
    SERVICE = 'services'
    INTRA = 'intra'
    types = (ALL, DEPARTMENT, SERVICE, INTRA)


class AutocompleteField(forms.ModelChoiceField):

    class Meta:
        model = None

    def __init__(self, **kw):
        kw['queryset'] = self.Meta.model.objects.all()
        super(AutocompleteField, self).__init__(**kw)
        self.widget = copy.deepcopy(self.widget)
        self._meta = self.Meta()
        if self.widget._model is None:
            self.widget._model = self._meta.model

    def clean(self, value):
        input_text = value[0]
        if input_text and input_text.strip():
            if value[1]:
                return super(AutocompleteField, self).clean(value[1])
            else:
                return self.clean_from_text(input_text)
        else:
            return super(AutocompleteField, self).clean(None)

    def clean_from_text(self, input_text):
        """
        This method being called when hidden input with unique id
        is absent, but text input is not empty. It probably means
        that user entered some data but has not selected a choice
        in autocomplete window. Subclasses should implement this
        to be able to `deduce` valid choice from chaotic input text.
        """
        return None

    def prepare_value(self, value):
        if isinstance(value, tuple):
            value = value[1]
        return super(AutocompleteField, self).prepare_value(value)


class StaffField(AutocompleteField):
    widget = AutocompleteWidget(attrs={'class': 'js-autocomplete_staff'})

    class Meta:
        model = Staff

    def clean_from_text(self, input_text):
        try:
            staff_obj = Staff.staff_manager.search_exactly(input_text)
        except (UserNotFound, UserFoundMultiple):
            raise forms.ValidationError(self.error_messages['invalid_choice'])
        else:
            return staff_obj


class StaffUserField(StaffField):

    widget = StaffUserAutocompleteWidget(
        attrs={'class': 'js-autocomplete_staff'})

    def clean(self, value):
        staff_obj = super(StaffUserField, self).clean(value)
        if staff_obj:
            return staff_obj.django_user
        else:
            return None


class GroupField(AutocompleteField):

    group_type = AUTOCOMPLETE_GROUP_TYPE.ALL

    def __init__(self, **kw):
        group_type = kw.pop('type', self.group_type)
        to_field_name = kw.get('to_field_name', 'id')
        # NOTE: maybe assert group_type in AUTOCOMPLETE_GROUP_TYPE.types?
        self.widget = AutocompleteWidget(
            attrs={
                'class': 'js-autocomplete_groups',
                'onclick': "return {hidden: '%s', type: '%s'}" %
                (to_field_name, group_type)
            }
        )
        super(GroupField, self).__init__(**kw)

    class Meta:
        model = Group


class DepartmentGroupField(GroupField):

    group_type = AUTOCOMPLETE_GROUP_TYPE.DEPARTMENT

    class Meta:
        model = Group


class DepartmentField(AutocompleteField):

    widget = AutocompleteWidget(attrs={'class': 'js-autocomplete_departments'})

    class Meta:
        model = Department


class I18nModelChoiceField(forms.ModelChoiceField):
    def label_from_instance(self, obj):
        return obj.i_name
