from json import loads
from django import forms


class FieldToDictConverter(object):

    def as_dict(self, form, name, field):
        full_name = 'field_%s_as_dict' % field.__class__.__name__
        if hasattr(self, full_name):
            return getattr(self, full_name)(form, name, field)
        return self._default_field_as_dict(form, name, field)

    def _default_field_as_dict(self, form, name, field):
        return {name: self._field_value(form, name, field)}

    def as_front_dict(self, form, name, field):
        full_name = 'field_%s_as_front_dict' % field.__class__.__name__
        if hasattr(self, full_name):
            return getattr(self, full_name)(form, name, field)
        return self._default_field_as_front_dict(form, name, field)

    def choices_as_front_dict(self, form, name, field):
        full_name = 'choices_%s_as_front_dict' % field.__class__.__name__
        if hasattr(self, full_name):
            return getattr(self, full_name)(form, name, field)
        return self._default_choices_as_front_dict(form, name, field)

    def choices_StaffLocalizedModelChoiceField_as_front_dict(self, form, name, field):
        return {
            name: [
                {'value': value, 'label': label}
                for value, label in field.choices
            ]
        }

    def choices_StaffPersonSuggestField_as_front_dict(self, form, name, field):
        return {}

    def choices_StaffChoiceField_as_front_dict(self, form, name, field):
        result = []
        if hasattr(field, 'choices'):
            for k, v in field.choices:
                result.append({'value': k, 'label': str(v)})
        return {name: result}

    def _default_choices_as_front_dict(self, form, name, field):
        if hasattr(field, 'choices'):
            return {name: [{'value': k, 'label': v} for k, v in field.choices]}

    def _default_field_as_front_dict(self, form, name, field):
        value_dict = {'value': self._field_value(form, name, field)}

        if hasattr(field, 'front_params'):
            value_dict.update({k: v for k, v in field.front_params.items()})

        return {name: value_dict}

    def field_StaffSuggestField_as_front_dict(self, form, name, field):

        return {
            name: {
                'value': self._field_value(form, name, field),
                'caption': form.initial.get(name + '_caption', ''),
            }
        }

    def field_StaffPersonSuggestField_as_front_dict(self, form, name, field):
        return self.field_StaffSuggestField_as_front_dict(form, name, field)

    def _field_value(self, form, name, field):
        # наполовину copypaste https://code.djangoproject.com/ticket/10427#comment:24
        if form.is_bound:
            if isinstance(field, forms.FileField) and field.data is None:
                val = form.initial.get(name, field.initial)
            else:
                val = form[name].value()
        else:
            val = form.initial.get(name, field.initial)
            if callable(val):
                val = val()
        if val is None:
            val = ''
        return val


class StaffForm(forms.Form):

    converter = FieldToDictConverter()
    strip_fields = []

    def choices_as_front_dict(self):
        return self._as_something(self.converter.choices_as_front_dict)

    def as_dict(self):
        return self._as_something(self.converter.as_dict)

    def as_front_dict(self):
        return self._as_something(self.converter.as_front_dict)

    def _as_something(self, f):
        result = {}
        for name, field in self.fields.items():
            r = f(self, name, field)
            if r is not None:
                result.update(r)
        return result

    def clean(self):
        super(StaffForm, self).clean()
        self._strip_fields()
        return self.cleaned_data

    def _strip_fields(self):
        for field in self.strip_fields:
            value = self.cleaned_data.get(field)
            if isinstance(value, str):
                self.cleaned_data[field] = value.strip()

    @property
    def errors(self):
        errors = dict(super(StaffForm, self).errors)
        for field, field_errors in errors.items():
            errors[field] = [loads(error) for error in field_errors]
        return errors
