from django.utils.translation import ugettext

from .fields import MulticField, MulticModelChoiceField


def field_json_adapter(bound_field, include_choices=True):
    """Преобразование поля джанго-формы в словарь."""

    field_dict = {
        'id': bound_field.auto_id,
        'errors': bound_field.errors,
        'help_text': bound_field.help_text,
        'initial': bound_field.field.initial,
        'is_hidden': bound_field.is_hidden,
        'label': ugettext(bound_field.label),
        'name': bound_field.name,
        'html_name': bound_field.html_name,
        'required': bound_field.field.required,
        'value': bound_field.value(),
    }

    if isinstance(bound_field.field, MulticField):
        names = [bound_field.name + '__type', bound_field.name + '__value']
        raw_value = bound_field.value()
        if isinstance(raw_value, (list, tuple)) and len(raw_value) == 3:
            values = raw_value
        else:
            values = bound_field.field.widget.decompress(raw_value)

        field_dict['multic_values'] = dict(zip(names, values[:2]))
        field_dict['value'] = values[2]

    if isinstance(bound_field.field, MulticModelChoiceField):
        field_name = bound_field.name
        field = bound_field.field
        value = bound_field.form.initial.get(field_name, field.initial)
        presentation = field.presentation_function(value) if value else ''

        field_dict['multic_values'] = {
            field_name + '__type': field.multic_type,
            field_name + '__value': presentation,
            }

    if include_choices and hasattr(bound_field.field, 'choices'):
        choices = [(str(first), str(second)) for
                   first, second in bound_field.field.choices]
        field_dict.update({
            'choices': choices
        })

    return field_dict


def form_json_adapter(form):
    """Чтобы отдавать форму с селектами для ChoiceField нужно явно перечислить
    поля с селектами в атрибуте формы ``include_choices_for``.
    Пример:
    class MyForm(forms.Form):
        offices = forms.ModelChoiceField(queryset=Office.objects.active())

        include_choices_for = ('offices',)

    """
    fields = {}
    include_choices_for = getattr(form, 'include_choices_for', [])
    for field in form:
        include_choices = field.name in include_choices_for
        fields[field.name] = field_json_adapter(field, include_choices=include_choices)

    return {
        'non_field_errors': form.non_field_errors(),
        'fields': fields,
    }
